/******************************************************************************
* 			 
*  FILE:	mreadout_pci.c
*  VERSION:	1.00
*  AUTHOR:	Jose Maria Panero
*  DATE:	07/10/2000
* 			 
*  DESCRIPTION:	This demo program opens a PCI device driver  and sends example
* 		commands to the controller. This demo is an upgrade of written
*		demo with the same name by Scott Streit an dated 01/10/2000. 
*
*  INPUT:	None
*
* 
*  REVISION HISTORY:
*	Date		Who	Version	Description
*  ---------------------------------------------------------------------------
* 	07/10/2000	jmp	1.00	Initial
* 			 
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <stropts.h>
#include "DSPCommand.h"
#include "LoadDspFile.h"
#include "FitsFile.h"
#include "Memory.h"
#include "Temperature.h"

#define PCI_DEV		"/dev/astropci0\0"
#define NCOLS		256
#define NROWS		256
#define TRUE		1
#define FALSE		0

/******** 
* menu options
********/ 
#define TDL_MENU_OPTION '1'		/*  test test data link		*/
#define WRM_MENU_OPTION '2'		/*  test write memory		*/
#define RDM_MENU_OPTION '3'		/*  test read memory		*/
#define LDA_MENU_OPTION '4'		/*  test load pci file		*/
#define SET_MENU_OPTION '5'		/*  test apply setup		*/
#define EXP_MENU_OPTION '6'		/*  test expose 		*/
#define NAM_MENU_OPTION '7'		/*  test enter fits file name	*/
#define FTS_MENU_OPTION '8'		/*  test write fits file	*/
#define CMD_MENU_OPTION '9'		/*  test manual command		*/
#define RES_MENU_OPTION 'a'		/*  test resset controller	*/
#define PON_MENU_OPTION 'b'		/*  test power on		*/
#define POF_MENU_OPTION 'c'		/*  test power off		*/
#define MSE_MENU_OPTION 'd'		/*  test modify setup		*/
#define MEX_MENU_OPTION 'e'		/*  test modify exposure time	*/
#define ALG_MENU_OPTION 'f'		/*  test set temperature alg.	*/
#define GTE_MENU_OPTION 'g'		/*  test get temperature	*/
#define SEE_MENU_OPTION 'h'		/*  test see image		*/
#define EXIT_MENU_OPTION '0'		/*  exit			*/

/******** 
* example struct to setup readout
********/ 
struct CameraSetup
{
	int do_reset;
	int do_tim_app;
	int do_tim_file;
	int do_util_app;
	int do_util_file;
	int do_power_on;
	int do_gain_and_speed;
	int do_idle;
	int do_dimensions;
	int do_open_shutter;
	int gain;
	int speed;
	int idle;
	int tim_app;
	int util_app;
	int cols;
	int rows;
	int exp_time;
};

char fits_filename[50]="Image.fts";	/* Initial fits filename	*/
char board[4][10]={"NO BOARD", "PCI_ID", "TIM_ID", "UTIL_ID"};
char result[50] = "";

/******** 
* extern variables
********/ 
extern int memory_type;

extern int reply_value;
extern char str_reply_value [];
extern int fits_reply;
extern char str_fits_reply [];

extern image_fd;			/* user's image buffer		*/

/******** 
* C function prototypes (all include in this file)
********/ 
char print_menu ();
int do_option (int pci_fd, char option);
int clear_screen ();
int please_enter_value (char *);
double please_enter_double (char *text);
int please_enter_int (char *text);
int enter_memory_type (char *);
char *enter_manual_command (char *text);

int test_test_data_link (int pci_fd) ;
int test_write_memory (int pci_fd) ;
int test_read_memory (int pci_fd) ;
int test_reset (int pci_fd);
int test_power_on (int pci_fd);
int test_power_off (int pci_fd);

int test_load_pci_file(int pci_fd);
int test_setup_controller(int pci_fd, struct CameraSetup cam_prop);
int expose(int pci_fd);

int test_modify_exp_time(int pci_fd, struct CameraSetup *cam_prop);
int test_modify_setup(int pci_fd, struct CameraSetup *cam_prop);

int test_enter_fits_filename ();
int test_write_fits_header (int pci_fd);

int test_set_temperature_algorithm ();
int test_get_temperature (int pci_fd);

int test_manual_command(int pci_fd);

int test_see_image (int pci_fd);


/******** 
* Global variables
********/ 
struct CameraSetup cam_prop = {
			FALSE, 	/* do reset		*/
			TRUE, 	/* do tim_app		*/ 
			FALSE, 	/* do tim_file		*/
			TRUE, 	/* do util_app		*/ 
			FALSE, 	/* do util_file		*/ 
			TRUE, 	/* do power_on		*/ 
			FALSE, 	/* do gain and speed	*/ 
			FALSE, 	/* do idle		*/ 
			TRUE, 	/* do dimensions	*/ 
			FALSE, 	/* do open shutter	*/ 
			1, 	/* gain			*/ 
			0,	/* speed		*/
			1, 	/* idle			*/ 
			0, 	/* tim_app		*/ 
			0, 	/* util_app		*/ 
			NCOLS, 	/* cols			*/ 
			NROWS, 	/* rows			*/
			1000	/* exposure time	*/
		};

int fd_out;		/*  file descriptor to fits file	*/


/****************************************************************************** 
*
*       Function:
*       ---------
*		int main()
*
*       Description:
*       ------------
*	Clear the screen, open the PCI device, create a memory buffer (user's
*	image buffer), and then launches the options menu, that lets the user
*	do the different tests that are implemented.
*	The menu loop asking the user for select  an option and executing it,
*	until the user selects the exit option.
*	Finally free the  allocated  memory for the user's image buffer, and
*	close the PCI device.
*
*       Parameters:
*       ---------
*	None.
*
*       Returns:
*       --------
*       Returns NO_ERROR.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/ 
int main()
{
	int pci_fd;		/*  file descriptor to the DSP device	*/
	int result;		/*  status after DSP command		*/
	char option;		/*  menu test, selected option		*/
	char text[80];		/*  comments				*/

	/******** 
	*  start.
	********/
	clear_screen ();
    	fprintf (stdout, "\nStart.\n");

	/******** 
	*  Try to open a device driver connection. If not successfull, exits
	*  the application.
	********/
 	if ((pci_fd=open(PCI_DEV, O_RDWR)) < 0 ) 
	{
    		sprintf(text, "Error. Could not open device driver \"%s\"", 
				PCI_DEV);
    		perror(text);
    		exit(EXIT_FAILURE);
    	} else {
    		fprintf (stdout, "Driver \"%s\" successfully opened.\n", 
			PCI_DEV);
	}

	/******** 
	*  The device driver connection was successfully established.
	*  This print out has been commented.
	fprintf (stdout, "\tpci_fd: 0x%08X\n", pci_fd);
	********/
	
	/******** 
	*  Create memory buffer for the user application.
	*
	*  There is also the option of create the buffer each time a readout
	*  is going to start (and free that buffer after when the write to
	*  fits file is done).
	*
	*  In this example, we create a memory buffer, and we use always the
	*  same buffer. Only, if the dimensions of the image are edited, the
	*  buffer is freed and generates a new buffer upon the dimensions.
	*
	********/
	create_memory (cam_prop.rows * cam_prop.cols * 2);
    	fprintf (stdout, "Image buffer allocated.(%dx%d)\n",
						cam_prop.rows, cam_prop.cols);

	/******** 
	* test
	********/
	option = EXIT_MENU_OPTION;
	option = print_menu ();
	while (option != EXIT_MENU_OPTION)
	{
		do_option (pci_fd, option);
		option = print_menu ();
	}


	/******** 
	* free memory.
	********/
	free_memory(image_fd);
    	fprintf (stdout, "Image buffer freed.\n");
	
	/******** 
	* Close the PCI board connection.
	********/
	close(pci_fd);
    	fprintf (stdout, "Driver \"%s\" closed.\n", PCI_DEV);

	/******** 
	* end
	********/
	fprintf(stdout, "Test end.\n\n\n");

	/******** 
	* exit.
	********/
	exit(NO_ERROR);
}
/******************************************************************************
*
*	End of Main. main() {}
*
******************************************************************************/

/******************************************************************************
*
*       Function:
*       ---------
*		int do_option (int pci_fd, char option)
*
*       Description:
*       ------------
*	Executes a menu option. The menu options are the different implemented
*	routines to test the use of the DSPCommand module.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*       option          The menu option was selected for the use, and thus
*			should be executed. 
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int do_option (int pci_fd, char option)
{
	int result= NO_ERROR;

	switch (option)
	{
		case TDL_MENU_OPTION:
			/* Test Data Link */
			result = test_test_data_link (pci_fd);
			break;

		case WRM_MENU_OPTION:
			/* Test Write Memory */ 
			result = test_write_memory (pci_fd);
			break;
	
		case RDM_MENU_OPTION:
			/* Test Read Memory */ 
			result = test_read_memory (pci_fd);
			break;

		case RES_MENU_OPTION:
			/* Test Reset Controller */ 
			result = test_reset (pci_fd);
			break;

		case PON_MENU_OPTION:
			/* Test Power On */ 
			result = test_power_on (pci_fd);
			break;

		case POF_MENU_OPTION:
			/* Test Power Off */ 
			result = test_power_off (pci_fd);
			break;

		case LDA_MENU_OPTION:
			/* Test Load Pci */ 
			result = test_load_pci_file (pci_fd);
			break;

		case SET_MENU_OPTION:
			/* Test Setup Controller */ 
			result = test_setup_controller(pci_fd, cam_prop);
			break;

		case MEX_MENU_OPTION:
			/* Test modify exposure time */ 
			result = test_modify_exp_time(pci_fd, &cam_prop);
			break;

		case MSE_MENU_OPTION:
			/* Test modify setup */ 
			result = test_modify_setup(pci_fd, &cam_prop);
			break;

		case EXP_MENU_OPTION:
			/* Take exposure */
			result = expose (pci_fd);
			break;

		case NAM_MENU_OPTION:
			/* Test enter the name of the fits file */
			result = test_enter_fits_filename ();
			break;

		case ALG_MENU_OPTION:
			/* Test set temperature algorithm */
			result = test_set_temperature_algorithm ();
			break;

		case GTE_MENU_OPTION:
			/* Test get temperature */
			result = test_get_temperature (pci_fd);
			break;

		case FTS_MENU_OPTION:
			/* write fits header */
			result = test_write_fits_header (pci_fd);
			break;

		case CMD_MENU_OPTION:
			/* test manual command */
			result = test_manual_command(pci_fd);
			break;

		case SEE_MENU_OPTION:
			/* test see image */
			result = test_see_image(pci_fd);
			break;

		default:
			break;
	}

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		char print_menu ()
*
*       Description:
*       ------------
*	Display in stdout  the menu with  the different  tests that are 
*	implemented. For reference, also display in stdout: the current
*	selected name of the fits file, the current selected dimensions
*	and exposure time. Then Prompt the user for enter an option. 
*	Return the char  entered for the  user when  selecting the menu 
*	option. If the user attempt to enter  an string (i.e. more than 
*	one character, only  the first  character is picked up  for the 
*	function and the rest are ignored.
*	
*       Parameters:
*       ---------
*	None
*
*       Returns:
*       --------
*       Returns the first character entered for the user that is suposed 
*	to be one of the menu options.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
char print_menu ()
{
	int result = ERROR;
	char option[80];

	fprintf (stdout, "\n  %s\n", 
"============================================================================");
	fprintf(stdout,"\tfits file: \"%s\", dimensions: %dcolsx%drows, %dms\n",
		fits_filename, cam_prop.cols, cam_prop.rows, cam_prop.exp_time);
	fprintf (stdout, "  %s\n", 
"============================================================================");

	fprintf (stdout, "\t\t\t%c.- Test Data Link\n", TDL_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Write Memory\n", WRM_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Read Memory\n", RDM_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Load pci file\n", LDA_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Apply controller Setup\n",SET_MENU_OPTION);
	fprintf (stdout, "\t\t\t%c.- Expose\n", EXP_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Enter fits file name\n", NAM_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Test write fits header\n",FTS_MENU_OPTION);
	fprintf (stdout, "\t\t\t%c.- Test Manual Command\n", CMD_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Reset Controller\n", RES_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Power on\n", PON_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Power off\n", POF_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Edit setup options\n", MSE_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Edit exposure time\n", MEX_MENU_OPTION);	
	fprintf (stdout, 
		"\t\t\t%c.- Set temperature Algorithm\n", ALG_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- Get temperature\n", GTE_MENU_OPTION);	
	fprintf (stdout, "\t\t\t%c.- See image\n", SEE_MENU_OPTION);	
	fprintf (stdout, "\n\t\t\t%c.- Exit\n", EXIT_MENU_OPTION);	

	fprintf (stdout, "\n\t\t\tEnter Option.....");	
	
	fflush (stdin);

	/***************
       	* Because it is impossible to tell without knowing the data in advance 
       	* how many characters gets() will read, and because gets() will continue
       	* to store characters past the end of the buffer, it is extremely 
       	* dangerous to use. It has been used to break computer security. 
	* Use fgets() instead.
	*
      	* char *fgets(char *s, int size, FILE *stream);
       	* fgets() reads in at most one less than size characters from stream and
       	* stores them into the buffer pointed to by s. Reading stops after an 
       	* EOF or a newline. If a newline is read, it is stored into the  buffer.
       	* A '\0' is stored after the last character in the buffer.
	***************/
	fgets (option, 3, stdin);

	fprintf (stdout, "\n");	
	fflush (stdin);

	return option[0];
}

/******************************************************************************
*
*       Function:
*       ---------
*		void clear_screen ()
*
*       Description:
*       ------------
*	Clear the screen.
*
*       Parameters:
*       ---------
*	None
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int clear_screen ()
{
	int result = NO_ERROR;

	result = system ("clear");

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_test_data_link (int pci_fd) 
*
*       Description:
*       ------------
*	Test the "test data link" (test_data_link) DSPCommand. Ask the user for a 
*	value and then, send the test data link command. Read the reply
*	and display that reply in stdout. 
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_test_data_link (int pci_fd) 
{
	int result=NO_ERROR;
	int test_value=0;
	int test_return=0;
	int test_board=1;

	fprintf (stdout, "\nTesting command, Test Data Link:\n");	

	/******** 
	* ask for the board ID#
	********/
	test_board=(please_enter_value(
		"\t\tPlease enter board ID# (1=PCI, 2=TIM, 3=UTIL): "));
	if ((test_board <1) || (test_board >3))
	{
		fprintf (stdout, "\t\tIncorrect board ID#.\n");	
		return ERROR;
	}

	/******** 
	* ask for the value to send in order to test the data link
	********/
	test_value =(please_enter_value ("\t\tEnter a data value: "));
	fprintf(stdout,"\tSend: 0x%08X to \"%s\"\n",
		test_value,board[test_board]);
		
	/******** 
	* set the board destination, check for errors and if no errors send
	* the test data link command.
	********/
	test_return = set_board_destination(pci_fd, TDL_ARG_NUM, test_board);
	if (test_return == DON)
	{
		test_return = test_data_link (pci_fd, test_value);
	}

	/******** 
	*  print out the results of the test.
	********/
	fprintf (stdout, "\t\"%s\". Receiving: 0x%08X\n", 
		board[test_board],test_return);
	fprintf (stdout, "\treply_value: 0x%08X", reply_value);
	fprintf (stdout, "\t%s\n\n", str_reply_value);

	/******** 
	* as a result return NO_ERROR or ERROR.
	********/
	if (test_return==ERROR)
	{
		result = ERROR;
	}

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_write_memory (int pci_fd) 
*
*       Description:
*       ------------
*	Test the "write memory" (write_memory) DSPCommand. Ask the user for a
*	board id#, a memory type, a memory address to write to, and a value to
*	for write there.
*	
*	Then, send the write memory command to the selected board and memory 
*	type.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_write_memory (int pci_fd) 
{
	int result=NO_ERROR;
	int reply_value=0;
	int test_return=0;
	int test_address=0;
	int test_data=0;
	int test_board=1;
	int test_memory_type=X;

	fprintf (stdout, "\nTesting command, Write Memory:\n");	

	/******** 
	* ask for the board ID#
	********/
	test_board=(please_enter_value(
		"\t\tPlease enter board ID# (1=PCI, 2=TIM, 3=UTIL): "));
	if ((test_board <1) || (test_board >3))
	{
		fprintf (stdout, "\t\tIncorrect board ID#.\n");	
		return ERROR;
	}

	/******** 
	* ask for the memory type
	********/
	test_memory_type=
		enter_memory_type("\t\tEnter Memory Type (X, Y, D, R, P): ");
	if ((test_memory_type == ERROR))
	{
		fprintf (stdout, "\t\tIncorrect memory type.\n");	
		fprintf (stdout, "\t\tCorrect values are X, Y, D, R, P.\n");
		return ERROR;
	}

	/******** 
	* ask for the memory address to where to write the data
	********/
	test_address =
		please_enter_value("\t\tMemory Address you want to write to: ");

	/******** 
	* ask for the data to write
	********/
	test_data= please_enter_value("\t\tValue that you want to write: ");

	/******** 
	* set the board destination, check for errors and if no errors set
	* the memory type.
	********/
	test_return = set_board_destination(pci_fd, WRM_ARG_NUM, test_board);
	if (test_return == DON)
	{
		test_return = set_type (test_memory_type);
		/******** 
		* send the write_memory command
		********/
		result = write_memory (pci_fd, test_address, test_data);
	}
	fprintf (stdout, 
	      "\tBoard: %s, mem_type: 0x%06X, address: 0x%06X, write: 0x%06X\n",
	      board[test_board], memory_type, test_address, test_data);

	/******** 
	*  print out the results of the test.
	********/
	fprintf (stdout, "\treply: 0x%08X, %s\n", reply_value, str_reply_value);

	/******** 
	* return the result 
	********/
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_read_memory (int pci_fd) 
*
*       Description:
*       ------------
*	Test the "read memory" (read_memory) DSPCommand. Ask the user for a 
*	board id#, a memory type and a memory address to read from.
*	Then send the read memory command to the board and memory type selected
*	and display the result in stdout.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_read_memory (int pci_fd) 
{
	int result=NO_ERROR;
	int reply_value=0;
	int test_return=0;
	int test_address=0;
	int test_data=0;
	int test_board=1;
	int test_memory_type=X;

	fprintf (stdout, "\nTesting command, Read Memory:\n");	

	/******** 
	* ask for the board ID#
	********/
	test_board=(please_enter_value(
		"\t\tPlease enter board ID# (1=PCI, 2=TIM, 3=UTIL): "));
	if ((test_board <1) || (test_board >3))
	{
		fprintf (stdout, "\t\tIncorrect board ID#.\n");	
		return ERROR;
	}

	/******** 
	* ask for the memory type
	********/
	test_memory_type=
		enter_memory_type("\t\tEnter Memory Type (X, Y, D, R, P): ");
	if ((test_memory_type == ERROR))
	{
		fprintf (stdout, "\t\tIncorrect memory type.\n");	
		fprintf (stdout, "\t\tCorrect values are X, Y, D, R, P.\n");
		return ERROR;
	}

	/******** 
	* ask for the memory address from where to read the data
	********/
	test_address =
	       please_enter_value("\t\tMemory Address you want to read from: ");

	/******** 
	* set the board destination, check for errors and if no errors set
	* the memory type.
	********/
	test_return = set_board_destination(pci_fd, RDM_ARG_NUM, PCI_ID);
	result = test_return;
	if (test_return == DON)
	{
		test_return = set_type (test_memory_type);
		/******** 
		* send the read_memory command
		********/
		test_data = read_memory (pci_fd, test_address);

		result = test_data;
	}

	/******** 
	*  print out the results of the test.
	********/
	fprintf (stdout, "\tBoard: %s, address: 0x%08X, read: 0x%08X\n", 
		board[test_board], test_address, test_data);
	fprintf (stdout, "\treply: 0x%08X, %s\n", reply_value, str_reply_value);

	/******** 
	* return the result
	********/
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_reset (int pci_fd) 
*
*       Description:
*       ------------
*	Test the "reset controller" on DSPCommand.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/04/2000
*
******************************************************************************/
int test_reset (int pci_fd) 
{
	int result=NO_ERROR;
	int test_return=0;

	fprintf (stdout, "\nTesting command, Reset Controller:\n");	

	test_return = reset_controller (pci_fd);

	fprintf (stdout, "\treply_value: 0x%08X", reply_value);
	fprintf (stdout, "\t%s\n", str_reply_value);

	if (test_return  == ERROR)
	{
		result = ERROR;
	} else {
		result = NO_ERROR;
	}

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_power_on (int pci_fd) 
*
*       Description:
*       ------------
*	Test the power on DSPCommand.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_power_on (int pci_fd) 
{
	int result=NO_ERROR;
	int test_return=0;

	fprintf (stdout, "\nTesting command, Power On:\n");	

	test_return = power_on(pci_fd);

	fprintf (stdout, "\treply_value: 0x%08X", reply_value);
	fprintf (stdout, "\t%s\n", str_reply_value);

	if (test_return  == ERROR)
	{
		result = ERROR;
	} else {
		result = NO_ERROR;
	}

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_power_off (int pci_fd) 
*
*       Description:
*       ------------
*	Test the power off DSPCommand.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_power_off (int pci_fd) 
{
	int result=NO_ERROR;
	int test_return=0;

	fprintf (stdout, "\nTesting command, Power Off:\n");	

	test_return = power_off (pci_fd);

	fprintf (stdout, "\treply_value: 0x%08X", reply_value);
	fprintf (stdout, "\t%s\n", str_reply_value);

	if (test_return  == ERROR)
	{
		result = ERROR;
	} else {
		result = NO_ERROR;
	}

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_load_pci_file(int pci_fd)
*
*       Description:
*       ------------
*	Test to load a "pci program file" to the pci board.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_load_pci_file(int pci_fd)
{
	int result = NO_ERROR;

	fprintf(stdout, "\tLoading pci file. By default loads \"pci.lod\"\n");
	result = load_pci_file(pci_fd, "pci.lod");
	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
					reply_value, str_reply_value);
	if (result == ERROR) 
	{
		fprintf ( stderr, "\t\tError when Loading Pci File.\n");
	}
	
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_setup_controller
*				(int pci_fd, struct CameraSetup cam_prop)
*
*       Description:
*       ------------
*	Test to apply a setup controller that consists in few DSPCommands
*	that are send to configure the controller.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*	cam_prop	CameraSetup which parameters are going to be edited.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_setup_controller(int pci_fd, struct CameraSetup cam_prop)
{
	int result = NO_ERROR;

	fprintf(stdout, "\nSetup controller.\n");

	/******** 
	* Reset the controller 
	********/
	if (cam_prop.do_reset) 
	{
		fprintf(stdout, "\tResetting the controller.\n");
		result = reset_controller(pci_fd);
		fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
					reply_value, str_reply_value);
		if (result == ERROR) 
		{
			fprintf ( stderr, 
					"\t\tError when Reset Controller.\n");
		}
	}
	
	/******** 
	* Load timing file 
	********/
	if (cam_prop.do_tim_file) 
	{
		fprintf(stdout, 
		       "\tLoading timing file. By default loads \"tim.lod\"\n");
		load_file(pci_fd, "tim.lod", "timing");
		fprintf ( stdout, "\t\treply_value: 0x%08X. %s\n", 
				reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf ( stderr, "\t\tError when Load Timing File.\n");
		}
	}
	
	/******** 
	* Load timing application 
	********/
	else if (cam_prop.do_tim_app) 
	{
		fprintf( stdout, "\tLoading timing application #%d.\n", 
							cam_prop.tim_app);
		result = load_application(pci_fd, TIM_ID, cam_prop.tim_app);
		fprintf ( stdout, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, 
				"\t\tError Loading Timing Application.\n");
		}
	}

	/******** 
	* Load the utility file 
	********/
	if (cam_prop.do_util_file) 
	{
		fprintf(stdout, 
		    "\tLoading utility file. By default loads \"util.lod\"\n ");
		load_file(pci_fd, "util.lod", "utility");	
		fprintf ( stdout, "\t\treply_value: 0x%08X. %s\n", 
				reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, 
			  "\t\tError Loading Utility Application from file.\n");
		}
	}

	/******** 
	* Load the utility application 
	********/
	else if (cam_prop.do_util_app) 
	{	
		fprintf(stdout, 
		      "\tLoading utility application #%d.\n",cam_prop.util_app);
		result = load_application(pci_fd, UTIL_ID, cam_prop.util_app);
		fprintf ( stdout, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, 
				"\t\tError Loading Utility Application.\n");
		}
	}
	
	/******** 
	* Do Power On 
	********/
	if (cam_prop.do_power_on) 
	{
		fprintf(stdout, "\tDoing power on.\n");
		result = power_on(pci_fd);
		fprintf ( stdout, "\t\treply_value: 0x%08X. %s\n", 
				reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, "\t\tError Doing Power On.\n");
		}
	}
	
	/******** 
	* Do gain and speed 
	********/
	if (cam_prop.do_gain_and_speed) 
	{
		fprintf(stdout, "\tSetting gain and speed.\n");
		result = set_gain(pci_fd, cam_prop.gain, cam_prop.speed);
		fprintf ( stdout, "\t\treply_value: 0x%08X. %s\n", 
				reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, "\t\tError Setting Gain & Speed.\n");
		}
	}
	
	/******** 
	* Do Idle 
	********/
	if (cam_prop.do_idle) 
	{
		fprintf(stdout, "\tDoing idle.\n");
		result = idle(pci_fd, cam_prop.idle);
		fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, "\t\tError Doing Idle.\n");
		}
	}
	
	/******** 
	* Set the array column dimension
	********/
	if (cam_prop.do_dimensions) 
	{
		fprintf(stdout, "\tSetting dimensions.\n");
		result = dimensions(pci_fd, cam_prop.rows, cam_prop.cols);
		fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
		if (result == ERROR) {
			fprintf( stderr, "\t\tError Setting up Dimensions.\n");
		}
	}
	
	/******** 
	* in any case return result.
	********/
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_modify_exp_time
*				(int pci_fd, struct CameraSetup *cam_prop)
*
*       Description:
*       ------------
*	Let the user modify some configuration parameters for the variable
*	"cam_prop" (configuration setup). The parameter is the exposure 
*	time of the image to be taken.
*	If the value entered for the exposure time is no valid (no numeric),
*	that exposure time is set to 0.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*	cam_prop	CameraSetup which parameters are going to be edited.
*
*       Returns:
*       --------
*       Returns NO_ERROR.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_modify_exp_time(int pci_fd, struct CameraSetup *cam_prop)
{
	int result = NO_ERROR;
	
	cam_prop->exp_time = ((int)
		please_enter_value ("\t\tNew Exposure Time: "));
	
	/******** 
	* in any case return result.
	********/
	return result;
}


/******************************************************************************
*
*       Function:
*       ---------
*		int test_modify_setup(int pci_fd, struct CameraSetup *cam_prop)
*
*       Description:
*       ------------
*	Let the user modify some configuration parameters for the variable
*	"cam_prop" (configuration setup). This parameters are the columns 
*	and rows (dimensions) of the image to be taken.
*	If the values entered for the dimensions are not valid (no numeric),
*	that dimension (rows or columns) is set to 0.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*	cam_prop	CameraSetup which parameters are going to be edited.
*
*       Returns:
*       --------
*       Returns NO_ERROR.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_modify_setup(int pci_fd, struct CameraSetup *cam_prop)
{
	int result = NO_ERROR;
	
	(cam_prop->cols) = ((int)
		please_enter_int ("\t\tNumber Columns: "));

	(cam_prop->rows) = ((int)
		please_enter_int ("\t\tNumber Rows: "));
	
	/******** 
	* The dimensions of the image may have changed, so that means the 
	* image buffer size could be outdated. To correct this, free  the
	* memory of the current image buffer and create a new one in base
	* of the new dimensions.
	* You also could check that, if the new buffer is going to be sma-
	* ller than the previous one, it's not neccessary to freed it and
	* create it again.
	********/

	/******** 
	* free memory.
	* the create_memory() function freed the buffer before allocate
	* a new one, so it's not neccessary do it manually.
	free_memory(image_fd);
    	fprintf (stdout, "\t\tImage buffer freed.\n");
	********/

	/******** 
	* At least should allocate space for one row and column.
	********/
	if (cam_prop->rows < 1)
	{
		cam_prop->rows = 1;
	}

	if (cam_prop->cols < 1)
	{
		cam_prop->cols = 1;
	}

	/******** 
	* create a new image buffer with the image dimensions.
	********/
	create_memory ((cam_prop->rows) * (cam_prop->cols) * 2);
    	fprintf (stdout, "\t\tImage buffer allocated.(%dx%d)\n",
						cam_prop->rows, cam_prop->cols);
    	fprintf (stdout, "\n\t\tNote: Image buffer size could have change.\n%s",
				"\t\tPlease, re-apply setup controller.\n");

	/******** 
	* in any case return result.
	********/
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int expose (int pci_fd)
*
*       Description:
*       ------------
*	Test the "expose" and "readout image" DSPCommands.
*	Use the global variable "cam_prop" to get the exposure time. Then,
*	set the exposure time and do expose. Readout the image to a buffer
*	and finally write the buffer (the image) to the fits file.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int expose (int pci_fd)
{
	int result = NO_ERROR;
	struct fits_header_struct fits_header;

	fprintf(stdout, "\nExpose.\n");

	/******** 
	* set shutter position.
	********/
	if (cam_prop.do_open_shutter) {
		fprintf( stdout, "\tShutter position: OPEN.\n");
	} else {
		fprintf( stdout, "\tShutter position: CLOSE.\n");
	}
	result= set_shutter_position(pci_fd, cam_prop.do_open_shutter);
	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);

	if (result == ERROR) {
		fprintf( stderr, "\t\tError setting up expose time.\n");
	}

	/******** 
	* set exposure time.
	********/
	fprintf( stdout, "\tSetting up expose time.\n");
	result = set_exposure_time(pci_fd, cam_prop.exp_time);
	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);

	if (result == ERROR) {
		fprintf( stderr, "\t\tError setting up expose time.\n");
	}

	/******** 
	* set dimensions.
	fprintf(stdout, "\tSetting up dimensions.\n");
	result = dimensions(pci_fd, cam_prop.rows, cam_prop.cols);
	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
					reply_value, str_reply_value);
	if (result == ERROR) {
		fprintf( stderr, "\t\tError Setting up Dimensions.\n");
	}
	********/
	
	/******** 
	* Start exposure.
	********/
	fprintf(stdout, "\tStart exposure.\n");
	/******** 
	result = set_board_destination(pci_fd, SEX_ARG_NUM, UTIL_ID);
	********/
	result = start_exposure(pci_fd);
	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
	if (result == ERROR) 
	{
		fprintf( stderr, "\t\tError when exposure.\n");
	}

	/******** 
	* Wait for exposure to end.
	if (cam_prop.exp_time >= EXP_TIME_LOW_LIMIT)
	{
		result = read_expose_time(pci_fd);
		fprintf (stdout,"\t\telapsed time=%d ms.\n", result);
		while ( result < cam_prop.exp_time - 1000)
		{
			sleep (1);
			result = read_expose_time(pci_fd);
			fprintf (stdout,"\t\telapsed time=%d ms.\n", result);
		}
		sleep (1);
	}
	********/
	
	/******** 
	* Start readout.
	********/
	fprintf(stdout, "\tStart readout.\n");

	result= read_image(pci_fd, cam_prop.rows * cam_prop.cols * 2, 
		cam_prop.exp_time, image_fd);
	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
	if (result == ERROR) 
	{
		fprintf( stderr, "\t\tError when read image.\n");
	}
	
	/******** 
	* swap data image
	swap_data (image_fd,  cam_prop.rows * cam_prop.cols * 2);
	********/

	/******** 
	* set shutter position to close.
	********/
	result= set_shutter_position(pci_fd, FALSE);

	/******** 
	* Write to fits file.
	********/
	fprintf (stdout, "\tWriting fits file.\n");

	set_fits_header (&fits_header);
	edit_fits_exposure_dimensions (&fits_header, 
			       cam_prop.rows, cam_prop.cols, cam_prop.exp_time);
	result = write_fits_file (fits_filename, image_fd, fits_header);

	fprintf ( stderr, "\t\t%s\n", str_fits_reply);
	if (result == ERROR) 
	{
		fprintf( stderr, "\t\tError when writing fits file.\n");
	}

	/******** 
	* in any case return result.
	********/
	return result;
}


/******************************************************************************
*
*       Function:
*       ---------
*		int test_write_fits_header (int pci_fd)
*
*       Description:
*       ------------
*	Simulate how to write to a fits file. No write no more than the
*	header.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_write_fits_header (int pci_fd)
{
	int result = NO_ERROR;

	struct fits_header_struct fits_header;

	set_fits_header (&fits_header);

	fd_out = open ("ftest.tst", O_RDWR | O_CREAT | O_TRUNC,	0666);
			
	write_fits_header (fd_out, fits_header);
	close (fd_out);

	return result;
}


/******************************************************************************
*
*       Function:
*       ---------
*		int test_manual_command(int pci_fd)
*
*       Description:
*       ------------
*	Test the DSPCommand "manual command". Ask the user for the arguments
*	(the board, the command id, and the specific arguments for that com-
*	mand.) Then send it to the PCI drive.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_manual_command(int pci_fd)
{
	int result = NO_ERROR;
	int board_id = TIM_ID;
	int command_id = SGN;
	int number_of_arguments;
	int arg_1 = ARG_NULL;
	int arg_2 = ARG_NULL;
	int arg_3 = ARG_NULL;
	int arg_4 = ARG_NULL;
	int arg_5 = ARG_NULL;

	fprintf(stdout, "\nTest Manual Command.\n");

	/******** 
	* ask for the board ID#
	********/
	board_id=(please_enter_value(
		"\t\tPlease enter board ID# (1=PCI, 2=TIM, 3=UTIL): "));
	if ((board_id <1) || (board_id >3))
	{
		fprintf (stdout, "\t\tIncorrect board ID#.\n");	
		return ERROR;
	}

	/******** 
	* ask for the command to be send
	********/
	command_id= to_value(enter_manual_command("\t\tCommand: "));
		
	if (command_id == ERROR)
	{
		fprintf (stdout, 
	  "\t\tIncorrect Command. Manual commands are 3 characters length.\n");	
		return ERROR;
	}

	/******** 
	* ask for the number of arguments
	********/
	number_of_arguments=(please_enter_value("\t\tnumber of arguments: "));
		
	if ((number_of_arguments <0) || (number_of_arguments >5))
	{
		fprintf (stdout, "\t\tIncorrect number of arguments. %s", 
			"Accept up to 5 arguments.\n");	
		return ERROR;
	}

	/******** 
	* ask for argument 1
	********/
	if (number_of_arguments > 0)
	{
		arg_1=(please_enter_value("\t\targument 1 [in hexadecimal]: "));
	}

	/******** 
	* ask for argument 2
	********/
	if (number_of_arguments > 1)
	{
		arg_2=(please_enter_value("\t\targument 2 [in hexadecimal]: "));
	}

	/******** 
	* ask for argument 3
	********/
	if (number_of_arguments > 2)
	{
		arg_3=(please_enter_value("\t\targument 3 [in hexadecimal]: "));
	}

	/******** 
	* ask for argument 4
	********/
	if (number_of_arguments > 3)
	{
		arg_4=(please_enter_value("\t\targument 4 [in hexadecimal]: "));
	}

	/******** 
	* ask for argument 5
	********/
	if (number_of_arguments > 4)
	{
		arg_5=(please_enter_value("\t\targument 5 [in hexadecimal]: "));
	}

	/******** 
	* tell the command is going to be send
	********/
	fprintf(stdout, "\t\tboard=0x%08X, ", board_id);
	fprintf(stdout, "command=0x%08X, \n", command_id);
	if (number_of_arguments > 0) fprintf(stdout,"\t\targ1=0x%08X  ", arg_1);
	if (number_of_arguments > 1) fprintf(stdout,"arg2=0x%08X  \n", arg_2);
	if (number_of_arguments > 2) fprintf(stdout,"\t\targ3=0x%08X  ", arg_3);
	if (number_of_arguments > 3) fprintf(stdout,"arg4=0x%08X  ", arg_4);
	if (number_of_arguments > 4) fprintf(stdout,"arg5=0x%08X  ", arg_5);
	fprintf(stdout, "\b\n");

	/******** 
	*
	* NOTE: when using the send_manual_command(), for null arguments use
	* 	ARG_NULL, do not use NULL.
	* 
	*	For example to send a two arguments command:
	*
	*	result= send_manual_command(pci_fd, board_id, command_id,
	*			arg_1, arg_2, ARG_NULL, ARG_NULL, ARG_NULL);
	*
	********/
	result= send_manual_command(pci_fd, board_id, command_id,
					arg_1, arg_2, arg_3, arg_4, arg_5);

	fprintf ( stderr, "\t\treply_value: 0x%08X. %s\n", 
						reply_value, str_reply_value);
	if (result == ERROR) 
	{
		fprintf( stderr, "\t\tError when sending manual command.\n");
	}

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_enter_fits_filename ()
*
*       Description:
*       ------------
*	Lets the user specify the name of the fits file. No errors are 
*	checked.
*
*       Parameters:
*       ---------
*	None
*
*       Returns:
*       --------
*	Returns NO_ERROR. No errors are checked.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_enter_fits_filename ()
{
	int result = NO_ERROR;

	fprintf(stdout,"\t\tcurrent fits file name is \"%s\".\n",fits_filename);
	fprintf (stdout,"\t\tPlease enter new name: ");
	fgets (fits_filename, 25, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	return result;
}


/******************************************************************************
*
*       Function:
*       ---------
*		int test_set_temperature_algorithm ()
*
*       Description:
*       ------------
*		Let the user to set the algorithm to use in order to convert
*	between temperature degrees and adu units. The user can select one of
*	the algorithms defined by the constants LINEAR and NONLINEAR in 
*	Temperature.h  -- "linear" or "nonlinear" --.
*
*		If the user selects the LINEAR algorithm, the function also
*	let the user to input a value for the two linear coefficients, 
*	coefficient 0 and coefficient 1.
*
*		If the user input an incorrect algorithm, the function prints
*	a message in stdout and returns ERROR.
*
*       Parameters:
*       -----------
*	None.
*
*       Returns:
*       --------
*       If the user input a not correct algorithm, returns ERROR, otherwise
*	returns NO_ERROR. Correct algorithms are defined by the constants 
*	LINEAR and NONLINEAR in Temperature.h
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/18/2000
*
******************************************************************************/
int test_set_temperature_algorithm ()
{
	int result = NO_ERROR;
	char new_algorithm[35];
	double coeff_0= (double)0.0;
	double coeff_1= (double)0.0;

	/******** 
	*  show the actual algorithm
	********/
	fprintf(stdout,"\t\tcurrent temperature algorithm is \"%s\".\n",
		get_temperature_algorithm());

	/******** 
	*  let the user input an algorithm
	********/
	fprintf (stdout,"\t\tPlease enter new algorithm (%s, %s): ",
		LINEAR, NONLINEAR);
	fgets (new_algorithm, 25, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	/******** 
	* cut the carriage return '\n' in the input
	********/
	if ( (strlen(new_algorithm)) > 0 );
		new_algorithm[strlen(new_algorithm)-1]='\0';

	/******** 
	*  set the algorithm. Note: there's not need to check for errors here
	*  because the set_algorithm() function does it.
	********/
	result = set_algorithm ( (char *) new_algorithm);

	/******** 
	*  case when the input is a not correct algorithm
	*  Returns.
	********/
	if (result == ERROR)
	{
		fprintf(stdout,
			"\t\tSorry, \"%s\" is not a correct algorithm.\n",
			new_algorithm);
		fprintf(stdout,
			"\t\tPlease use one of:\n\t\t\t\t\t%s\n\t\t\t\t\t%s\n", 
			LINEAR, NONLINEAR); 
		return result;
	}

	/******** 
	*  show the actual algorithm
	********/
	fprintf(stdout,"\t\tcurrent temperature algorithm is \"%s\".\n",
		get_temperature_algorithm());

	/******** 
	*  if the actual algorithm is LINEAR let the user to input the
	*  linear coefficients.
	********/
	if ( strcmp (get_temperature_algorithm(), LINEAR) == 0)
	{
		/******** 
		*  first show the actual linear coefficients value
		********/
		fprintf(stdout,"\t\tcurrent linear coefficient 0 is % #9.4f\n",
			get_temperature_linear_coeff_0());
	       fprintf(stdout,"\t\tcurrent linear coefficient 1 is % #9.4f\n\n",
			get_temperature_linear_coeff_1());

		/******** 
		*  then let the user input new linear coefficients value
		*  If the input is not correct, the value returned by
		*  please_enter_double() is (double)0.0
		********/
		coeff_0= (please_enter_double("\t\tenter coefficient 0: "));
		coeff_1= (please_enter_double("\t\tenter coefficient 1: "));

		/******** 
		*  set the linear coefficients value
		********/
		set_linear_coefficients (coeff_0, coeff_1);

		/******** 
		*  finally show the actual algorithm and 
		*  linear coefficients value
		********/
		fprintf(stdout,"\t\tcurrent temperature algorithm is \"%s\".\n",
			get_temperature_algorithm());
		fprintf(stdout,"\t\tcurrent linear coefficient 0 is % #9.4f\n",
			get_temperature_linear_coeff_0());
		fprintf(stdout,"\t\tcurrent linear coefficient 1 is % #9.4f\n",
			get_temperature_linear_coeff_1());
	}
	
	/******** 
	*  return the result
	********/
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_get_temperature (int pci_fd)
*
*       Description:
*       ------------
*	Test the DSPCommand get_tenperature. Send the command and then
*	display the value returned.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns NO_ERROR.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_get_temperature (int pci_fd)
{
	int result = NO_ERROR;
	double temperature;

	/******** 
	*  show the actual algorithm
	********/
	fprintf(stdout,"\t\tcurrent temperature algorithm is \"%s\".\n",
		get_temperature_algorithm());

	/******** 
	*  if the actual algorithm is LINEAR show the
	*  linear coefficients.
	********/
	if ( strcmp (get_temperature_algorithm(), LINEAR) == 0)
	{
		fprintf(stdout,"\t\tcurrent linear coefficient 0 is % #9.4f\n",
			get_temperature_linear_coeff_0());
		fprintf(stdout,"\t\tcurrent linear coefficient 1 is % #9.4f\n",
			get_temperature_linear_coeff_1());
	}

	/******** 
	*  get temperature
	********/
	temperature = get_temperature (pci_fd);

	/******** 
	*  show temperature
	********/
	fprintf(stdout,"\t\ttemperature=\"% #6.1f\".\n",temperature);

	/******** 
	*  return the result
	********/
	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int test_see_image (int pci_fd)
*
*       Description:
*       ------------
*	Sends a command to the SystemCommand in order to open the saoimage
*	application to display a fits image. The file that will try to dis-
*	play is that one with file name the current value of fits_filename.
*
*       Parameters:
*       ---------
*       pci_fd          Pointer to the opened PCI device.
*
*       Returns:
*       --------
*       Returns ERROR if not successfull.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int test_see_image (int pci_fd)
{
	int result = NO_ERROR;
	char command[80];

	fprintf(stdout, "\t\tSending image file \"%s\" to saoimage application",
			fits_filename);
	sprintf (command, "saoimage -fits %s &", fits_filename);
	result = system (command);

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int please_enter_value (char *text)
*
*       Description:
*       ------------
*	Prompt the user for to enter a numerical value. The value will be taken
*  as a base 16 value (hexadecimal).
*
*       Parameters:
*       ---------
*	text	a text string to write as a prompt
*
*       Returns:
*       --------
*	A int value which contains the value entered by the user in base 16
*	(hexadecimal).
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
******************************************************************************/
int please_enter_value (char *text)
{
	char result[50] = "0x0";
	int hexa_result;
	
	fprintf (stdout,"%s", text);
	fgets (result, 50, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	hexa_result = strtol (result, NULL, 16);
	return (hexa_result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		double please_enter_double (char *text)
*
*       Description:
*       ------------
*	Prompt the user for to enter a numerical value. The value will be taken
*  	as a double.
*
*       Parameters:
*       ---------
*	text	a text string to write as a prompt
*
*       Returns:
*       --------
*	A double value which contains the value entered by the user.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/18/2000
*
******************************************************************************/
double please_enter_double (char *text)
{
	char result[50] = "0.0";
	double double_result;
	
	fprintf (stdout,"%s", text);
	fgets (result, 50, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	double_result = atof (result);
	return (double_result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int please_enter_int (char *text)
*
*       Description:
*       ------------
*	Prompt the user for to enter a numerical value. The value will be taken
*  	as an int.
*
*       Parameters:
*       ---------
*	text	a text string to write as a prompt
*
*       Returns:
*       --------
*	A int value which contains the value entered by the user.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/18/2000
*
******************************************************************************/
int please_enter_int (char *text)
{
	char result[50] = "0";
	int int_result;
	
	fprintf (stdout,"%s", text);
	fgets (result, 50, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	int_result = atoi (result);
	return (int_result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int enter_memory_type (char *text)
/******************************************************************************
*
*       Function:
*       ---------
*		int enter_memory_type (char *text)
*
*       Description:
*       ------------
*	Prompt the user for to enter a memory type. Memory type could be: 
*  X, Y, D, R or P. See DSPCommand.h
*
*       Parameters:
*       ---------
*	text	a text string to write as a prompt
*
*       Returns:
*       --------
*	A int value which contains the memory type selected by the user, or
*  ERROR, if there is not selection or a not known memory type.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/07/2000
*
******************************************************************************/
int enter_memory_type (char *text)
{
	char result[50] = "0x0";
	
	fprintf (stdout,"%s", text);
	fgets (result, 50, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	/******** 
	* check for the user enter only a character
	********/
	if ((strlen(result) > 2) || (strlen(result) < 1))
	{
		return ERROR;
	}

	/******** 
	* check for the memory type: X, Y, D, R, P
	********/
	if ((result[0]=='X') ||  (result[0]=='x'))
	{
		return X;
	} else
	if ((result[0]=='Y') ||  (result[0]=='y'))
	{
		return Y;
	} else
	if ((result[0]=='D') ||  (result[0]=='d'))
	{
		return D;
	} else
	if ((result[0]=='R') ||  (result[0]=='r'))
	{
		return R;
	} else
	if ((result[0]=='P') ||  (result[0]=='p'))
	{
		return P;
	}

	/******** 
	* if arrive here means the user had enter another value than X, Y, D,
	* R, P, so will be return ERROR
	********/
	return ERROR;
}

/******************************************************************************
*
*       Function:
*       ---------
*		char *enter_manual_command (char *text)
*
*       Description:
*       ------------
*	Prompt the user for to enter a string value. 
*
*       Parameters:
*       ---------
*	text	a text string to write as a prompt
*
*       Returns:
*       --------
*	A string value which is the value entered by the user.
*
*       Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/11/2000
*
******************************************************************************/
char *enter_manual_command (char *text)
{
	
	fprintf (stdout,"%s", text);
	fgets (result, 50, stdin);
	fflush (stdin);
	fprintf (stdout,"\t\t\n");

	/******** 
	* cut the carriage return '\n' in the input
	********/
	result[strlen(result)-1]='\0';

	return (result);
}
/******************************************************************************
*
*		END OF CODE 	meadout_pci.c
*
******************************************************************************/
