/******************************************************************************
*
*   File:       Memory.c
*   Version:    1.00
*   Author:     Jose Maria Panero Ciprian
*   Abstract:   C library functions for memory allocation and de-allocation.
*
*
*   Revision History:
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/00        jmp     1.00    Initial
*
******************************************************************************/

#include <stdio.h>		/*  for stderr usage			*/
#include <errno.h>


int image_fd;			/*********
				* image_fd will be a pointer to the last chunk
				* of memory allocated. 
				*
				* 	The only meaning is for the user  that
				* can use directly  this variable image_fd. In
				* this case, image_fd should be declared in the
				* user code as a extern. (see:)
				*
				*	extern int image_fd;
				*
				*********/

/*******************************************************************************
*
*	Function:
*	---------
*	int create_memory(int image_size)
*
*	Description:
*	------------
*       This function is used to allocate space to an image buffer.
*	The function returns a pointer to a block of at least 'image_size'
*	bytes suitably aligned for any use.
*
*	Parameters:     
*	-----------
*	image_size  	The size of the memory buffer (in bytes) to be
*           		allocated.
*
*       Returns:        
*	--------
*	A reference to the image buffer is returned. 
*	Error is set if the image memory buffers cannot be allocated.
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*	Revision History:
*	-----------------
*	Date            Who   Version    Description
*	----------------------------------------------------------------------
*	07/10/2000      jmp     1.00    Initial
*
*                       
*******************************************************************************/
int create_memory(int image_size)
{
        unsigned char *fd;
	int *mem_ptr;
	

	/************ 
	* The next is to allow only a memory image buffer at once, in order
	* to prevent stack overflow as a result to allocate too many buffer.
	* There is supose that we only work (in memory) with one image at a
	* time. 
	************/ 
	mem_ptr=(void *)image_fd;

        if (mem_ptr != NULL) 
	{
                free(mem_ptr);
        }

	/************ 
        * Create the image buffer.
	************/ 
        fd = (unsigned char *)malloc(image_size);

	/************ 
        * Check for colorfull memory errors.
	*
     	* The malloc() function will fail if:
	*
     	*	ENOMEM	The physical limits of the system are exceeded  by
	*		size  bytes of memory which cannot be allocated.
	*
	*	EAGAIN	There is not enough memory available at this point
	*		in time to allocate size  bytes of memory; but the
	*		application could try again later.
	************/ 
        if (fd == NULL && errno == ENOMEM) 
	{
  		fprintf (stderr, "Error Allocating Memory image buffers. %s",
			"System's physical limits exceded.");
        }

        else if (fd == NULL && errno == EAGAIN) 
	{
  		fprintf (stderr, "Error Allocating Memory image buffers. %s",
			"Not enough memory available. \n");
        }

        /**********
	*  Check for any other errors.
        **********/
        else if (fd == NULL) 
	{
		fprintf (stderr, 
			"Error Allocating Memory (errno = %d) \n", errno);
        }

        /**********
	*  update the image_fd
        **********/
	image_fd = (int)fd;

        /**********
	*  return the image_fd
        **********/
        return image_fd;
}


/*******************************************************************************
*
*	Function:
*	---------
*	int free_memory(int mem_fd)
*
*	Description:
*	------------
*       This function is used to de-allocate the space of an image buffer.
*	The argument to the free_memory function is a pointer to a memory block 
*	previously allocated.
*	This space is made available for further allocation after the function
*	is executed. If  mem_fd is a null pointer, no action occurs.
*	Undefined results will occur if some  random number is passed to the
*	function as a parameter.
*
*	Parameters:     
*	-----------
*	mem_fd  	Pointer to a memory block previously allocated.
*
*       Returns:        
*	--------
*	By default returns the global variable 'image_fd' that is a file
*	descriptor to the user's image buffer, that after the execution of
*	free_memory(), is suposed to be null.
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*	Revision History:
*	-----------------
*	Date            Who   Version    Description
*	----------------------------------------------------------------------
*	07/10/2000      jmp     1.00    Initial
*
*******************************************************************************/
int free_memory(int mem_fd)
{
	int *mem_ptr;

	mem_ptr=(void *)mem_fd;

        if (mem_ptr != NULL) 
	{
                free(mem_ptr);
        }

        /**********
	*  by default return the image_fd
        **********/
        return image_fd;
}


/******************************************************************************
*
*               END OF CODE. MODULE    Memory.c
*
******************************************************************************/
