/********************************************************************************
*   File:       LoadDspFile.c
*   Version:    1.00
*   Author:     Jose Maria Panero Ciprian
*
*   Abstract:   C library function for load DSP files.
*
*		The LoadDspFile adds the funcionality to the DSPCommand.c 
*		to load files into the boards, pci board, util board,  or 
*		timing board.
*
*		Two C functions,  load_file() used to load a  file either 
*		to the util board or to the timing board. 
*		And, load_pci_file() to load a file to the pci board.
*
*		Note: that this module may be  used with the DSPCommand.c
*		      module.	
*
*   Revision History:
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/00        jmp     1.00    Initial. The module is the result of
*					merge the modules LoadDspFile.c, and
*					PCIBoot.c, author  Scott Streit. The 
*					functions are updated to support the
*					global vars:
*					reply_value and str_reply_value
*					(see functions for details).
*
*
*******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "DSPCommand.h"
#include "LoadDspFile.h"


/*******************************************************************************
*       General use variables (from the DSPCommand).
*******************************************************************************/
extern int reply_value;
extern char str_reply_value[];

/*******************************************************************************
*
*	Function:
*	---------
*		int load_file(int pci_fd, const char *filename, 
*						const char *expected_file_type)
*
* 	Description:	
*	------------
*	This function loads a file into the DSP of the specified controller 
*	board. The controller board can be the  timing board or the utility
*	board as defined by the parameter expected_file_type.
*
*
*       Parameters:
*       -----------
*       pci_fd 			File descriptor for the pci device.
*	filename 		The name of the file to load.
*	expected_file_type 	The controller board to load the file into.
*				May be "timing" or "utility".
* 				This parameter is not case sensitive.
*
*	Returns:
*	--------
*	If successfull returns NO_ERROR, and if fail returns as ERROR the
*	source of the fail.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated accordingly.
*
*	Version: 	1.00
*	Author:  	Scott Streit
* 	Date:		11/22/1999
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/1999	sds     1.00    Initial
*
*	07/10/2000	jmp		Add support for the error variables
*					reply_value, str_reply_value
*					Take out comments a print outs and
*					resend them to the error variables.
*					Fits the funcionality of the function
*					accordingly the DSPCommand module.
*
*******************************************************************************/
int load_file(int pci_fd, const char *filename, const char *expected_file_type)
{
	int result = NO_ERROR;

	const char * input_value = " ";
	char input_line[81];
	const char * type_string = "__";
	const char * addr_string = " ";
	const char *data_string = " ";
	const char * file_type = " ";
	const char * path = "./";
	const char * name = " ";
	int end_of_file = 0;
	int done_reading_data = 0;
	int type = 0;
	int addr = 0;
	int data = 0;
	int n = -1;
	long file_position = 0;
	FILE *inFile;
	int reply_value = 0;
	int count = 0;
	int i = 0;
	const char *tokens[20];
	
	/****** 
	fprintf(stdout, "\t\tLoad file \"%s\"\n", filename);
	******/
		
	/****** 
	* Open the file for reading.
	******/
	if ((inFile = fopen(filename, "r")) == NULL) 
	{
		/****** 
		fprintf(stderr, "Error: Cannot open file: %s \n", filename);
		******/
		reply_value = ERR;
		sprintf (str_reply_value, 
				"Error: Cannot open file: %s \n", filename);
		result = ERROR;
		return result;
	}
		
	/****** 
	* Read in the file one line at a time.
	******/
	while (end_of_file == 0) 
	{
		fgets(input_line, 80, inFile);

		input_value = strtok(input_line, " ");
			
		/****** 
		* Check for the start of valid data.
		******/
		if (strcmp(input_value, "_DATA") == 0) {
			type_string = strtok(NULL, " ");
			addr_string = strtok(NULL, " ");

			/****** 
			* Convert the "address" string to the correct 
			* data type.
			******/
			sscanf(addr_string, "%X", &addr);
				
			/****** 
			* A valid start address must be less than 
			* MAX_DSP_START_LOAD_ADDR. 
			******/
			if (addr < MAX_DSP_START_LOAD_ADDR) {
				/****** 
				fprintf(stdout, 
					"\n Setting the memory type: %s \n", 
					type_string);
				******/
					
				/****** 
				* Set the "type" string to the correct 
				* ASCII sequence.
				******/
				if (strcmp(type_string, "X") == 0)
					set_type(X);
				else if (strcmp(type_string, "Y") == 0)
					set_type(Y);
				else if (strcmp(type_string, "P") == 0)
					set_type(P);
				else if (strcmp(type_string, "R") == 0)
					set_type(R);
				else if (strcmp(type_string, "D") == 0)
					set_type(D);
					
				/******* 
				*  Read in the data block. 
				*******/ 
				while (!done_reading_data) 
				{
					fgets(input_line, 80, inFile);
					count = 0;
					tokens[count] = strtok(input_line, " ");
						
					if (strstr(tokens[0], "_") != NULL) 
					{
						done_reading_data = 1;
						fseek (inFile, -15, SEEK_CUR);
					}
						
					else 
					{
						while (tokens[count] != NULL) 
						{
							count++;
							tokens[count] = 
							     strtok(NULL, " ");
						}
										
						for (i=0; i<count-1; i++) 
						{
							sscanf(tokens[i], "%X",
 									&data);
							/*******
							*  print out data.
			fprintf(stdout,"addr=0x%08X  data=0x%08X", addr, data);
							*******/
							result = write_memory (pci_fd, 
								    addr, data);
							/*******
							* if there is ERROR 
							* after write_memory, exits.
							*******/
							if (result==ERROR)
							{
								return result;
							}
							addr++;
							/*******
							* to print out in the 
							*  same place on the 
							* screen
							for (j=0; j<32; j++)
							   fprintf(stdout,"\b");
							*******/
						}
					}
				}
			}
		}
			
		/****** 
		* Check the file validity and set the board destination.
		******/ 
		if (strcmp(input_value, "_START") == 0) 
		{
			file_type = strtok(NULL, " ");
				
			if ((strcmp(file_type, "TIMBOOT") == 0) && 
			    (strcmp(expected_file_type, "timing") == 0)) 
			{
				result = set_board_destination(pci_fd, 
							WRM_ARG_NUM, TIM_ID);
				/*******
				* if there is ERROR after set_board_destination,
				* exits.
				*******/
				if (result==ERROR)
				{
					return result;
				}
			}
			
			else if ((strcmp(file_type, "UTILBOOT") == 0) && 
				 (strcmp(expected_file_type, "utility") == 0)) 
			{
				result = set_board_destination(pci_fd, 
							WRM_ARG_NUM, UTIL_ID);
				/*******
				* if there is ERROR after set_board_destination,
				* exits.
				*******/
				if (result==ERROR)
				{
					return result;
				}
			}
			else
			{
				reply_value = ERR;
				sprintf (str_reply_value, 
					"Error Unexpected file type.");
				result = ERROR;
				return result;
			}
		}
			
		/******
		* Check for the end of file.
		******/
		else if (strcmp(input_value, "_END") == 0) 
		{
			end_of_file = 1;
		}
			
		/******
		* Check for the end of file.
		******/
		else if (strcmp(input_value, "_SYMBOL") == 0) 
		{
			end_of_file = 1;
		}
						
		/******
		* Re-initialize variables. 
		******/
		done_reading_data = 0;
		type_string = "__";
		input_value = NULL;
	}
		
	/******
	* Close the file. 
	******/
	fclose(inFile);

	/******
	* Return result. 
	******/
	reply_value = DON;
	sprintf (str_reply_value, "DON, %s file \"%s\" loaded.", 
			expected_file_type, filename );

	return result;
}

/*******************************************************************************
*
*	Function:
*	---------
*		int load_pci_file(int pci_fd, const char *filename)
*
* 	Description:	
*	------------
*	This function downloads PCI boot code from a file to the PCI board
* 	DSP program memory space (P). The file download must  be performed
* 	before any  other  controller  actions occur,  otherwise, the  PCI
* 	board's actions are undefined.
*			
*
*       Parameters:
*       -----------
*       pci_fd 		File descriptor for the pci device.
*	filename 	The name of the file to load, e.g. "pci.lod"
*
*	Returns:
*	--------
*	If successfull returns NO_ERROR, and if fail returns as ERROR the
*	source of the fail.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated accordingly.
*
*
*       Version:	1.00
*       Author:	Scott Streit
* 	Date:		02/01/2000
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
* 	02/01/2000	sds	1.00	Initial
*
* 	02/11/2000	sds		Updated to check that the total number
*					of words to transfer is less than MAX_ 
* 					and that the start addr equals 0.
*
*	07/10/2000	jmp		Add support for the error variables
*					reply_value, str_reply_value. 
*					Change the order of checks. First of 
*					all check if the pci load file can be
*					open, before send any command to the
*					HTF register, otherwise the pci device
*					state can became undefinied.
*					Take out comments a print outs and
*					resend them to the error variables.
*					Fits the funcionality of the function
*					accordingly the DSPCommand module.
*
******************************************************************************/
int load_pci_file(int pci_fd, const char *filename)
{
	int result = NO_ERROR;

	char input_line[81];
	const char *words_string = " ";
	const char *addr_string = " ";
	const char *data_string = " ";
	const char *tokens[20];
	int data = 0;
	int addr = 0;
	int end_of_file = 0;
	int done_reading_data = 0;
	int reply = 0;
	int word_total = 0;
	int word_count = 0;
	int count = 0;
	int i = 0;
	int j = 0;
	long file_position = 0;
	FILE *inFile;
		
	/********** 
	* Open the file for reading.
	**********/ 
	if ((inFile = fopen(filename, "r")) == NULL) 
	{
		reply_value = ERR;
		sprintf (str_reply_value, 
				"Error: Cannot open file: \"%s\".", filename);
		result = ERROR;
		return result;
	}
		
	/********** 
	* Set the pci to download mode.
	**********/ 
	reply = set_pci_to_download_mode(pci_fd);	

	if (reply== ERROR)
	{
		/********** 
		* error recovering:
		* try the best the let the board in a stable mode.
		**********/ 
		reply = set_pci_to_transfer_mode(pci_fd);

		reply_value = ERR;
		sprintf (str_reply_value, 
				"Error: Cannot set pci to download mode.");
		result = ERROR;
		return result;
	}

	/********** 
	* main loop to read data.
	**********/ 
	while(!done_reading_data) 
	{
		fgets(input_line, 80, inFile);
		
		if (strstr(input_line, SEARCH_STRING) != NULL) 
		{
			/********** 
			* Get the next line. 
			**********/ 
			fgets(input_line, 80, inFile);
				
			/********** 
			* Get the number of words and starting address.
			**********/ 
			words_string = strtok(input_line, " ");
			addr_string = strtok(NULL, " ");
				
			sscanf(words_string, "%X", &word_total);
			sscanf(addr_string, "%X", &addr);
			
			/********** 
			* Check number of words is less MAX_WORD_NUM_PCI_FILE
			**********/ 
			if (word_total > MAX_WORD_NUM_PCI_FILE) 
			{
				reply_value = ERR;
				sprintf (str_reply_value, 
				  "Error: Number of words to write from \"%s\"",
				  filename);
				sprintf (str_reply_value, 
					"%s exceeds DSP memory range (%d).", 
					str_reply_value, 
					MAX_WORD_NUM_PCI_FILE);
				result = ERROR;
				return result;
			}
			/********** 
			* write the number of words in the argument1 register
			**********/ 
			else 
			{
				reply = 
				download_mode_set_argument1(pci_fd, word_total);
				/*******
				* if there is ERROR after set arg1 register,
				* exits.
				*******/
				if (reply==ERROR)
				{
					return reply;
				}
			}

			/********** 
			* Check starting address equal to LOAD_PCI_START_ADDR. 
			**********/ 
			if (addr != LOAD_PCI_START_ADDR) 
			{
				reply_value = ERR;
				sprintf (str_reply_value, 
				  	"Error: Starting address from \"%s\"",
				  	filename);
				sprintf (str_reply_value, 
					"%s is not equal to zero.", 
					str_reply_value); 
				result = ERROR;
				return result;
			}
			/********** 
			* write the start address in the argument1 register
			**********/ 
			else 
			{
			     	reply = 
				     download_mode_set_argument1(pci_fd, addr);
				/*******
				* if there is ERROR after set arg1 register,
				* exits.
				*******/
				if (reply==ERROR)
				{
					return reply;
				}
			     	addr = LOAD_PCI_START_ADDR;
			}

			/********** 
			* Throw away next line (e.g.: _DATA P 000002).
			**********/ 
			fgets(input_line, 80, inFile);
				
			/********** 
			* Load the data. 
			**********/ 
			while (word_count < (word_total - 2)) 
			{
				/********** 
				* Get the next line, that's the data start.
				**********/ 
				fgets(input_line, 80, inFile);

				/********** 
				* Check for "_DATA" strings and discard them by
				* reading  the next data line, which  should be
				* data. 
				**********/ 
				if (strstr(input_line, SEARCH_STRING) != NULL)
					fgets(input_line, 80, inFile);
				
				count = 0;
				tokens[count] = strtok(input_line, " ");
					
				if (strstr(tokens[0], "_") != NULL) 
				{
					done_reading_data = 1;
					fseek (inFile, -15, SEEK_CUR);
				}
						
				else 
				{
					while (tokens[count] != NULL) 
					{
						count++;
						tokens[count]=strtok(NULL," ");
					}
					
					for (i=0; i<count-1; i++) 
					{
						sscanf(tokens[i], "%X", &data);
						/********** 
				printf("addr= 0x%06X  data= 0x%06X",addr,data);
						**********/ 

	 			reply=download_mode_set_argument1(pci_fd,data);
						/*******
						* if there is ERROR after set 
						* arg1 register, exits.
						*******/
						if (reply==ERROR)
						{
							return reply;
						}
						addr++;
						word_count++;
						
						/********** 
						* to print out in the same
						* place on the screen.
						for (j=0; j<32; j++)
							fprintf(stdout, "\b");
						**********/ 
					}
				}
			}
			done_reading_data = 1;
		}
	}

	/********** 
	* Set the PCI board to data transfer mode. 
	**********/ 
	reply = set_pci_to_transfer_mode(pci_fd);

	if (reply== ERROR)
	{
		reply_value = ERR;
		sprintf (str_reply_value, 
				"Error: Cannot set pci to transfer mode.");
		result = ERROR;
		return result;
	}
	
	/********** 
	* Close the file and flush the reply buffer
	**********/ 
	fclose(inFile);
	flush_reply_buffer(pci_fd);

	/******
	* Return result. 
	******/
	reply_value = reply;
	sprintf (str_reply_value, "DON. Pci file \"%s\" loaded.", filename);

	return result;

}


/******************************************************************************
*
*               END OF CODE     LoadDspFile.c
*
******************************************************************************/
