/******************************************************************************
*
*   File:       FitsFile.c
*   Version:    1.00
*   Author:     Jose Maria Panero Ciprian
*   Abstract:   C library for fits-file functions.
*
*
*   Revision History:
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/00        jmp     1.00 	Initial
*
******************************************************************************/

#include <stdio.h>		/*  for SEEK_SET			*/
#include <fcntl.h>		/*  for O_RDWR, O_CREAT, O_TRUNC, ...	*/
#include <sys/mman.h>		/*  for memory management, PROT_READ...	*/
#include <time.h>		/*  get the date for the file's header	*/

#include "FitsFile.h"		/*  FitsFile module include file	*/

/*******************************************************************************
*	General use variables.
*	These var, fits_reply, and str_fits_reply, are used to keep the 
*  status of the fits-file commands. 
*  	Can be used in other modules/programs (instiated as a extern var)
*  to check the result of a fits-file module function. 
*
*	Example of use:
*
*	extern int fits_reply;
*	extern char *str_fits_reply;
*	int result;
*	
*	result = write_fits_file (filename, image_fd, fits_header_struct) 
*	fprintf (stdout, "fits reply= %d, %s", fits_reply, str_fits_reply);
*
*******************************************************************************/
int  fits_reply;
char str_fits_reply [80];	


/*******************************************************************************
*
*       Function:
*       ---------
*	int write_fits_file (const char *filename, int image_fd, 
*					struct fits_header_struct fits_header)
*
*       Description:
*       ------------
*	Writes in the file designed by filename, the contents of the buffer
*	pointed by image_fd that is the user's image buffer. And, adds as a
*	header of that file the fits_strcut that is a header.
*
*	Fits file are files that have a special header, that describes its 
*	contents and another characteristics.
*
*	This function open the file with name filename (if exits overwrites 
*	and, if doesn't creates it), adds the header and,  append the image
*	data that it's stored in the image_fd.
*
*	If there is some input/output problem related with the file I/O 
*	operation, the function returns ERROR. The variables fits_reply 
*	and str_fits_reply, are updated with the result of the function
*	either successfull or fail. 
*
*       Parameters:
*       -----------
*       filename                The name of the file where to save the fits
*				image.
*	image_fd		memory buffer that contains the fits image.
*	fits_header		header for the fits file.
*
*	Returns:
*	--------
*	The result of to write the image buffer to the file, either success
*	or fail.
*
*	Other effects:
*	--------------
*	fits_reply and str_fits_reply are updated.
*	If success, fits_reply is the number of written bytes.
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*   	Revision History:
*   	-----------------
*       Date            Who   Version    Description
*   	----------------------------------------------------------------------
*       07/10/2000      jmp     1.00    Initial
*
*
*******************************************************************************/
int write_fits_file (const char *filename, int image_fd, 
					struct fits_header_struct fits_header)
{
	int result = NO_ERROR;	/* return value.			     */
	int fd_out;		/* file descriptor to the file to write to.  */
	int file_size;		/* so far the size of the fits file in bytes */
	int header_size;	/* so far the size allowed to the header.    */
	int image_size;		/* so far the size of the image in bytes     */
	char *out_addr;		/* memory map for the fits file.	     */
	int i;			/* loop purpouses.			     */
	int columns = 0;	/* columns of the image			     */
	int rows = 0;		/* rows of the image			     */
	int nccds = 1;
	int bytes_per_pixel=0;	/* image's bytes per pixel.		     */
	int num_bytes_to_fill=0;/* number of bytes to fill with 0's in order
				   to complete a block.		     	     */


	/**********
	*  columns and rows of the image.
	**********/
	columns = fits_header.naxis1;
	rows = fits_header.naxis2;
	bytes_per_pixel = (int)(fits_header.bitpix / BITS_PER_BYTE);

	/**********
	*  open the file in readwrite mode.
	*  if some, error, returns it.
	**********/
	fd_out = open (filename, O_RDWR | O_CREAT | O_TRUNC, 0666);
	if (fd_out < 0 )
	{
		result = ERROR;
		fits_reply = fd_out;
		sprintf(str_fits_reply,"Can't open the file \"%s\".", filename);
		return result;
	}

	/**********
	*  Advance the size of the file.
	**********/
	header_size = FITS_BLOCK_SIZE;
	image_size = ( columns * rows * nccds * bytes_per_pixel);
	num_bytes_to_fill = FITS_BLOCK_SIZE - (image_size % FITS_BLOCK_SIZE);
	file_size = header_size + image_size + num_bytes_to_fill;

	/**********
	*  Advise the file size to get system buffer allocation. Write a "1" 
	*  at the end of the file. So if the file exits will be overwritten.
	**********/
	lseek (fd_out, (file_size-1), SEEK_SET);
	write (fd_out, "", 1);

	/**********
	*  positioning to the begining of the file.
	**********/
	lseek (fd_out, 0, SEEK_SET);
	write_fits_header (fd_out, fits_header);

	/**********
	*  memory map the file.
	**********/
	out_addr = mmap (0, file_size, PROT_READ | PROT_WRITE, MAP_SHARED,
				fd_out, 0);

	/**********
	*  copy the image in the memory location into the file memory map.
	**********/
	for (i=0; i < image_size; i++)
	{
		*(out_addr + i + FITS_BLOCK_SIZE) = *((caddr_t)image_fd + i);
	}

	/**********
	*  fill with FILL_DATA value, in order to complete a block.
	**********/
	for (i=0; i < num_bytes_to_fill; i++)
	{
		*(out_addr + FITS_BLOCK_SIZE + image_size + i) = FILL_DATA;
	}

	/**********
	*  memory unmap the file.
	**********/
	munmap (out_addr, file_size);
	
	/**********
	*  close the fits file.
	**********/
	close (fd_out);

	/**********
	*  update fits_reply and str_fits_reply
	**********/
	result = NO_ERROR;
	fits_reply = file_size;
	sprintf(str_fits_reply,"Fits file \"%s\" have been written (%d bytes).",
			filename, file_size);

	/**********
	*  return result
	**********/
	return result;
}


/*******************************************************************************
*
*       Function:
*       ---------
*	int write_fits_header(int fd_out, struct fits_header_struct fits_header)
*
*       Description:
*       ------------
*	Writes in the file designed by fd_out the corresponding header.
*	The header is a parameter. To fill the header see the functions 
*	below:
*
*	Fits file are files that have a special header, that describes its 
*	contents and another characteristics.
*
*       Parameters:
*       -----------
*       fd_out 		file descriptor to the file where to write to.	
*	fits_header	header for the fits file.
*
*	Returns:
*	--------
*	The result of to write the header to the file.
*
*	Other effects:
*	--------------
*	fits_reply and str_fits_reply are updated.
*	If success, fits_reply is the number of propierties that the header
*	has.
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*   	Revision History:
*   	-----------------
*       Date            Who   Version    Description
*   	----------------------------------------------------------------------
*       07/10/2000      jmp     1.00    Initial
*
*
*******************************************************************************/
int write_fits_header (int fd_out, struct fits_header_struct fits_header)
{
	int result = NO_ERROR;			/*  return value	    */
	char card[MAX_FITS_HEADER_CARDS][81];	/*  fits file header array  */
	int card_counter =0;			/*  number of propierties 
						    in the header	    */

	/**********
	*  Gets the actual date and time.
	**********/
	set_fits_header_time (&fits_header);

	/**********
	*  fill the fits file header array.
	**********/
        sprintf (card[card_counter++],  "SIMPLE  =           %10s                                                  ", fits_header.simple);
        sprintf (card[card_counter++],  "BITPIX  =           %10d                                                  ", fits_header.bitpix);
        sprintf (card[card_counter++],  "NAXIS   =           %10d                                                  ", fits_header.naxis);
        sprintf (card[card_counter++],  "NAXIS1  =           %10d                                                  ", fits_header.naxis1);
        sprintf (card[card_counter++],  "NAXIS2  =           %10d                                                  ", fits_header.naxis2);
        sprintf (card[card_counter++],  "DATE    =           %10s                                                  ", fits_header.date);
        sprintf (card[card_counter++],  "TIME    = %20s                                                  ", fits_header.time);
        sprintf (card[card_counter++],  "LOCATION= %20s                                                  ", fits_header.location);
        sprintf (card[card_counter++],  "SIDETIME= %20s                                                  ", fits_header.sidetime);
        sprintf (card[card_counter++],  "EPOCH   = %20s                                                  ", fits_header.epoch);
        sprintf (card[card_counter++],  "AIRMASS = %20s                                                  ", fits_header.airmass);
        sprintf (card[card_counter++],  "EXP_TIME=           %10d                                                  ", fits_header.exp_time);
        sprintf (card[card_counter++],  "IMG_TYPE= %20s                                                  ", fits_header.img_type);
        sprintf (card[card_counter++],  "TELESCOP= %20s                                                  ", fits_header.telescope);
        sprintf (card[card_counter++],  "INSTRMEN= %20s                                                  ", fits_header.instrment);
        sprintf (card[card_counter++],  "FILTER  = %20s                                                  ", fits_header.filter);
        sprintf (card[card_counter++],  "OBJECT  = %20s                                                  ", fits_header.object);
        sprintf (card[card_counter++],  "RA      = %20s                                                  ", fits_header.ra);
        sprintf (card[card_counter++],  "DEC     = %20s                                                  ", fits_header.dec);
        sprintf (card[card_counter++],  "OBSERVER= %20s                                                  ", fits_header.observer);
        sprintf (card[card_counter++],  "COMMENT1= %20s                                                  ", fits_header.comment1);
        sprintf (card[card_counter++],  "COMMENT2= %20s                                                  ", fits_header.comment2);
        sprintf (card[card_counter++],  "COMMENT3= %20s                                                  ", fits_header.comment3);
        sprintf (card[card_counter++],  "COMMENT4= %20s                                                  ", fits_header.comment4);
        sprintf (card[card_counter++],  "END%77s", " ");

	/**********
	*  update fits_reply
	**********/
	fits_reply = card_counter;

	/**********
	*  complete the fits file header array, up to MAX_FITS_HEADER_CARDS,
	*  with empty cards.
	**********/
	while (card_counter < MAX_FITS_HEADER_CARDS)
	{
        	sprintf (card[card_counter++],  "%80s", " ");
	}

	/**********
	*  write the fits file header array, to the file
	**********/
	card_counter = 0;
	while (card_counter < MAX_FITS_HEADER_CARDS)
	{
		write (fd_out, card[card_counter++], 80);
	}

	/**********
	*  update str_fits_reply
	**********/
	result = NO_ERROR;
	sprintf(str_fits_reply,"Fits file header have been written.");

	/**********
	*  return result
	**********/
	return result;
}


/*******************************************************************************
*
*       Function:
*       ---------
*	int set_fits_header (struct fits_header_struct *fits_header)
*
*       Description:
*       ------------
*	Sets the values of the header to default values.
*
*       Parameters:
*       -----------
*	fits_header	header to be set.
*
*	Returns:
*	--------
*	The result of to set the header to default values.
*
*	Other effects:
*	--------------
*	fits_reply and str_fits_reply are updated.
*	fits_reply has the same value that the function returns. 
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*   	Revision History:
*   	-----------------
*       Date            Who   Version    Description
*   	----------------------------------------------------------------------
*       07/10/2000      jmp     1.00    Initial
*
*
*******************************************************************************/
int set_fits_header (struct fits_header_struct *fits_header)
{
	int result = NO_ERROR;

	sprintf (fits_header->simple, HEADER_DEFAULT_SIMPLE);
	fits_header->bitpix = HEADER_DEFAULT_BITPIX;
	fits_header->naxis = HEADER_DEFAULT_NAXIS;
	fits_header->naxis1 = 0;
	fits_header->naxis2 = 0;
	sprintf (fits_header->date, "mm/dd/yyyy");
	sprintf (fits_header->time, "hh:mm:ss");
	sprintf (fits_header->location, "Observatory");
	sprintf (fits_header->sidetime, "hh:mm:ss.ss");
	sprintf (fits_header->epoch, "cccc");
	sprintf (fits_header->airmass, "99.99");
	fits_header->exp_time = 0;
	sprintf (fits_header->img_type, "Image");
	sprintf (fits_header->telescope, "Telescope");
	sprintf (fits_header->instrment, "Instrument");
	sprintf (fits_header->filter, "None");
	sprintf (fits_header->object, "None");
	sprintf (fits_header->ra, "hh:mm:ss.ss");
	sprintf (fits_header->dec, "hh:mm:ss.ss");
	sprintf (fits_header->observer, "Observer");
	sprintf (fits_header->comment1, "None");
	sprintf (fits_header->comment2, "None");
	sprintf (fits_header->comment3, "None");
	sprintf (fits_header->comment4, "None");

	/**********
	*  update fits_reply and str_fits_reply
	**********/
	result = NO_ERROR;
	fits_reply = result;
	sprintf(str_fits_reply,"Fits file header have been set.");

	/**********
	*  any case, result is returned
	**********/
	return result;
}


/*******************************************************************************
*
*       Function:
*       ---------
*	int edit_fits_exposure_dimensions(
*	struct fits_header_struct *fits_header, int cols,int rows, int exp_time)
*
*       Description:
*       ------------
*	Writes in the designed header, the values corresponding to the
*	image's number of rows, number of columns, and exposure time.
*
*
*       Parameters:
*       -----------
*	fits_header	header for the fits file, and where the propierties
*			are going to be written.
*	rows		number of rows.
*	cols		number of columns.
*	exp_time	exposure time.
*
*	Returns:
*	--------
*	The result of to write to the header the specified propierties. 
*
*	Other effects:
*	--------------
*	fits_reply and str_fits_reply are updated.
*	fits_reply has the same value that the function returns. 
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*   	Revision History:
*   	-----------------
*       Date            Who   Version    Description
*   	----------------------------------------------------------------------
*       07/10/2000      jmp     1.00    Initial
*
*
*******************************************************************************/
int edit_fits_exposure_dimensions (struct fits_header_struct *fits_header, 
					int rows, int cols, int exp_time)
{
	int result = NO_ERROR;

        fits_header->naxis1 = cols;
        fits_header->naxis2 = rows;
        fits_header->exp_time = exp_time;

	/**********
	*  update fits_reply and str_fits_reply
	**********/
	result = NO_ERROR;
	fits_reply = result;
	sprintf(str_fits_reply,
		"Fits file header have been updated (%dx%d, %dms).",
        	fits_header->naxis1,
        	fits_header->naxis2,
        	fits_header->exp_time);

	/**********
	*  any case, result is returned
	**********/
	return result;
}


/*******************************************************************************
*
*       Function:
*       ---------
*	int set_fits_header_time (struct fits_header_struct *fits_header)
*
*       Description:
*       ------------
*	Writes in the designed header, the values corresponding to the
*	date, time and epoch.
*
*
*       Parameters:
*       -----------
*	fits_header	header for the fits file, and where the propierties
*			are going to be written.
*
*	Returns:
*	--------
*	The result of to write to the header the specified propierties. 
*
*	Other effects:
*	--------------
*	fits_reply and str_fits_reply are updated.
*	fits_reply has the same value that the function returns. 
*
*
*       Version:        1.00
*       Author:         Jose Maria Panero Ciprian
*       Date:           07/10/2000
*
*   	Revision History:
*   	-----------------
*       Date            Who   Version    Description
*   	----------------------------------------------------------------------
*       07/10/2000      jmp     1.00    Initial
*
*******************************************************************************/
int set_fits_header_time (struct fits_header_struct *fits_header)
{
	int result = NO_ERROR;
	time_t time_now;
	struct tm *now;

	time_now = time (NULL);
	now = localtime(&time_now);

	sprintf (fits_header->date, "%02d/%02d/%04d", now->tm_mon + 1,
							now->tm_mday,
							now->tm_year + 1900);
	sprintf (fits_header->time, "%02d:%02d:%02d.00", now->tm_hour,
							now->tm_min,
							now->tm_sec);
	sprintf (fits_header->epoch, "%04d", now->tm_year + 1900);

	/**********
	*  update str_fits_reply
	**********/
	result = NO_ERROR;
	fits_reply = result;
	sprintf(str_fits_reply,
		"Fits file header date and time have been updated (%s %s).",
		fits_header->date,
		fits_header->time);

	/**********
	*  any case, result is returned
	**********/
	return result;
}


/******************************************************************************
*
*               END OF CODE. MODULE    FitsFile.c
*
******************************************************************************/
