/******************************************************************************
*
*   File:       DSPCommand.h
*   Version:	1.00
*   Author:     Scott Streit
*   Abstract:   C library for DSP functions.
*		I/O control commands for Solaris device driver for the SDSU PCI
*               Interface Card, and DSP Commands.
*               These commands are for use with ioctl().
*
*
*   Revision History:
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*	11/22/99	sds	1.00	Initial
*
*       06/20/00        jmp     1.00 	A bunch of values are added. Below, 
*					there are some:
*					Added the constants:
*						NONE
*						CLEARED_REPLY_VALUE
*                         	    		ASTROPCI_ABORT_READ
*                    	          		ASTROPCI_FLUSH_REPLY_BUFFER
*						READ_CONTROLLER_STATUS
*						WRITE_CONTROLLER_STATUS	
*						_OPEN_SHUTTER_
*						_CLOSE_SHUTTER_	
*						EXP_TIME_LOW_LIMIT
*						ARG_NULL
*						TDL_ARG_NUM, RDM_ARG_NUM, ....
*
*					These replies are added :
*						EMTY, TOUT, RST
*
*					Added control:
*						#ifndef __ASTRO_IO_H__
*
*					Added sections:
*						"Memory Location Id Constants"
*  						"Masks to set the HCTR"
*
*					Added the section under the control:
*						#ifndef __boolean__
*
*					Added the C function prototypes :
*						power_off ()
*
*						set_exposure_time ()
*						pause_exposure ()
*						resume_exposure ()
*
*						abort_readout ()
*
*						set_shutter_position()
*						set_temperature()
*
*						flush_reply_buffer()
*						check_error_reply_value ()
*
*						set_pci_to_download_mode()
*						set_pci_to_transfer_mode()
*						download_mode_set_argument1()
*
*						get_hctr_register()
*						set_hctr_register()
*
*						send_manual_command()
*
******************************************************************************/

#ifndef __boolean__
#define __boolean__
#define FALSE			0
#define TRUE			1
#define UNDEFINED		-1
#define ERROR			-1
#define NO_ERROR		0
#endif
/*********************************************
*	end of the #ifndef __boolean__
*********************************************/

#ifndef DSPCOMMAND_H
#define DSPCOMMAND_H

#define	EXP_TIME_LOW_LIMIT	5001
#define	ARG_NULL		0xFFFFFF
#define	INT_NULL		(int)0x0

/******************************************************************************
*	Argument number for DSP Commands 
******************************************************************************/
#define  TDL_ARG_NUM    	(int)0x00010000
#define  RDM_ARG_NUM    	(int)0x00020000
#define  WRM_ARG_NUM    	(int)0x00030000
#define  PON_ARG_NUM    	(int)0x00000000
#define  SEX_ARG_NUM    	(int)0x00000000
#define  EXP_ARG_NUM    	(int)0x00020000

#define  TIM_APP_ARG_NUM        (int)0x00010000
#define  UTIL_APP_ARG_NUM       (int)0x00010000

/******************************************************************************
*	DSP Command Constants
******************************************************************************/
#define READ_CONTROLLER_STATUS	0x8079
#define WRITE_CONTROLLER_STATUS	0x807B
#define RESET_CONTROLLER	0x807D
#define LOAD_APPLICATION	0x807F
#define MANUAL_COMMAND 		0x8081
#define READ_PCI_STATUS 	0x8083
#define TEST_DATA_LINK		0x8085
#define READ_MEMORY  		0x8087
#define WRITE_MEMORY  		0x8089
#define PCI_DOWNLOAD  		0x808B
#define POWER_ON		0x808D
#define POWER_OFF		0x808F
#define SET_BIAS_VOLTAGES	0x8091
#define CLEAR_ARRAY 		0x8093
#define STOP_IDLE_MODE		0x8095
#define RESUME_IDLE_MODE  	0x8097
#define READ_EXPOSURE_TIME	0x8099
#define START_EXPOSURE 		0x809B
#define READ_IMAGE     		0x809D
#define ABORT_READOUT   	0x809F
#define ABORT_EXPOSURE 		0x80A1
#define PAUSE_EXPOSURE		0x80A3
#define RESUME_EXPOSURE		0x80A5
#define RESERVED_1     		0x80A7
#define OPEN_SHUTTER   		0x80A9
#define CLOSE_SHUTTER		0x80AB
#define _OPEN_SHUTTER_		1
#define _CLOSE_SHUTTER_		0

#define SET_ARRAY_TEMPERATURE 	0x80AD
#define READ_ARRAY_TEMPERATURE 	0x80AF
/*************************************************
*        Define replies
*************************************************/
#define EMTY			0x454D5459
#define TOUT			0x544F5554
#define DON                  	0x00444F4E
#define ERR			0x00455252
#define SYR               	0x00535952
#define SGN			0x0053474E
#define RST			0x00525354
#define NONE                    0x4E4F4E45
#define CLEARED_REPLY_VALUE     -1
#define MAX_WAIT_TIME		1	/* Max wait time in seconds */
#define _OPEN_SHUTTER 		1
#define _CLOSE_SHUTTER          0

/******************************************************************************
*	Board Id Constants
******************************************************************************/
#define PCI_ID			1
#define TIM_ID			2
#define UTIL_ID			3

/******************************************************************************
*	Memory Location Id Constants
*		__D	DRAM
*		__R	ROM
*		__P	DSP program memory space
*		__X	DSP X memory space
*		__Y	DSP Y memory space
******************************************************************************/
#define D                       0x5f5f44        /*  Ascii __D  */
#define R                       0x5f5f52        /*  Ascii __R  */
#define P                       0x5f5f50        /*  Ascii __P  */
#define X                       0x5f5f58        /*  Ascii __X  */
#define Y                       0x5f5f59        /*  Ascii __Y  */

/******************************************************************************
*        DDC  DEVICE DRIVER COMMANDS Constants
******************************************************************************/
#ifndef __ASTRO_IO_H__
#define __ASTRO_IO_H__

/*************************************************
*        DDC,  Information retrieval commands
*************************************************/
#define ASTROPCI_GET_CMDR		0x0
#define ASTROPCI_GET_REPLY		0x3
#define ASTROPCI_GET_HCTR		0x5
#define ASTROPCI_GET_HSTR		0x6
#define ASTROPCI_GET_PROGRESS		0x7
#define ASTROPCI_GET_CONFIG_INFO        0x314

/*************************************************
*        DDC,  Information set commands
*************************************************/
#define ASTROPCI_SET_CMDR  		0x100
#define ASTROPCI_SET_ARG1 	        0x105
#define ASTROPCI_SET_ARG2 	        0x106
#define ASTROPCI_SET_ARG3 	        0x107
#define ASTROPCI_SET_ARG4 	        0x108
#define ASTROPCI_SET_ARG5 	        0x109
#define ASTROPCI_SET_ARGS 	        0x110
#define ASTROPCI_SET_DESTINATION	0x111
#define ASTROPCI_SET_HCTR 	        0x115
#define ASTROPCI_SET_HCVR 	        0x117
#define ASTROPCI_SET_EXPTIME 	        0x119
#define ASTROPCI_SET_NCOLS 	        0x120
#define ASTROPCI_SET_NROWS 	        0x121
#define ASTROPCI_SET_IMAGE_BUFFERS	0x122
#define ASTROPCI_SET_UTIL_OPTIONS	0x123
#define ASTROPCI_SET_BINNING_NCOLS	0x125
#define ASTROPCI_SET_BINNING_NROWS	0x126
#define ASTROPCI_CLEAR_REPLY		0x301
#define ASTROPCI_ABORT_READ             0x302
#define ASTROPCI_FLUSH_REPLY_BUFFER     0x124

#endif
/*********************************************
*	end of the #ifndef  __ASTRO_IO_H__
*********************************************/

/*********************************************
*  Masks to set the Host Control Register HCTR.
*
*       Only three bits of this register are used. Two are control bits to set
*  the mode of the PCI board (bits 8 and 9)  and  the  other (bit 3) is a flag
*  indicating the progress of image data transfer to the user's application.
*
*       Bit 3   = 1     Image buffer busy transferring to user space.
*               = 0     Image buffer not  transferring to user space.
*
*       Bit 8= 0 & Bit 9= 1   PCI board set to slave mode for PCI file download.
*       Bit 8= 0 & Bit 9= 0   PCI board set to normal processing.
*
*       Note that the HTF_MASK, sets the HTF bits 8 and 9 to transfer mode.
*
*********************************************/
#define HTF_MASK        0x200
#define HTF_CLEAR_MASK  0xFFFFFCFF
#define BIT3_CLEAR_MASK 0xFFFFFFF7
#define BIT3_SET_MASK   0x00000008
#define SET_MODE_DONE   0x00000038

/******************************************************************************
*	Function Prototypes
******************************************************************************/
extern int set_board_destination(
		int pci_fd, int number_of_args, int board_destination);
extern int set_type(int type);

extern int test_data_link (int pci_fd, int data);
extern int read_memory (int pci_fd, int address);
extern int write_memory (int pci_fd, int address, int data);
extern int pon (int pci_fd);
extern int power_off (int pci_fd);

extern int set_hcvr(int pci_id, int command, int expected_reply);
extern int arg(int pci_id, int arg_number, int arg_value);
extern int cmdr(int pci_fd, int command);

extern int start_exposure(int pci_fd);
extern int read_image(int pci_fd, int byte_count,int exposure_time, int mem_fd);
extern int dimensions(int pci_fd, int ccd_rows, int ccd_columns);

extern int read_expose_time(int pci_fd);
extern int set_exposure_time(int pci_fd, int exposure_time);
extern int abort_exposure(int pci_fd);
extern int pause_exposure(int pci_fd);
extern int resume_exposure(int pci_fd);

extern int abort_readout(int pci_fd);

extern int reset_controller(int pci_fd);

extern int open_shutter(int pci_fd);
extern int close_shutter(int pci_fd);
extern int set_shutter_position(int pci_fd, int do_open_shutter);
extern int shutter_position(int pci_fd, int do_open_shutter);

extern int idle(int pci_fd, int mode);

extern int set_gain (int pci_fd, int gain, int speed);

extern int load_application(int pci_fd, int board_id, int app_number);
extern int set_temperature(int pci_fd, double temperature);
extern double get_temperature(int pci_fd);

extern int clear_reply(int pci_fd);
extern int read_reply (int pci_fd, int expected_value);
extern int flush_reply_buffer(int pci_fd);

extern int set_pci_to_download_mode(int pci_fd);
extern int set_pci_to_transfer_mode(int pci_fd);
extern int download_mode_set_argument1(int pci_fd, int value);

extern int get_hctr_register(int pci_fd);
extern int set_hctr_register(int pci_fd, int value);

extern int send_manual_command(int pci_fd, int board_id, int command_id,
                        int arg_1, int arg_2, int arg_3, int arg_4, int arg_5);

extern int check_error_reply_value (int error);

extern char *to_ascii(int value);
extern int to_value(char *text);


#endif
/*********************************************
*	end of the #ifndef DSPCOMMAND_H
*********************************************/
/******************************************************************************
*
*               END OF CODE. FILE    DSPCommand.h
*
******************************************************************************/
