/******************************************************************************
*
*   File:       DSPCommand.c
*   Version:    1.00
*   Author:     Scott Streit
*   Abstract:   C library for DSP functions.
*
*
*   Revision History:
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    
*					Added support for global variables:
*						reply_value, str_reply_value,
*						arg_and_board_dest.
*					Update functions and add new ones.
*					(see functions).
*
******************************************************************************/

#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <math.h>
#include "DSPCommand.h"		/*  DSPCommand module include file	*/
#include "Temperature.h"	/*  for get and set the temperature	*/



/*******************************************************************************
*
*       General use variables.
*
*       These var, like reply_value, and str_reply_value, are used to keep the
*  status of the DSP commands.
*       Can be used in other modules/programs (instiated as a extern var)
*  to check the result of a DSP command.
*
*       Example of use:
*
*       extern int reply_value;
*       extern char *str_reply_value;
*       int result;
*
*       result = tdl (pci_fd, data)
*       fprintf (stdout, "reply= %d, %s", reply_value, str_reply_value);
*
*******************************************************************************/
int arg_and_board_dest;
int memory_type;
int reply_value;
char str_reply_value[125]="Clear";

/*********************
*  auxiliar variable for to use in the auxiliar function to_ascii()
*********************/
char ascii_value[3];

/******************************************************************************
*
*       Function:
*       ---------
*		int set_board_destination (
*			int pci_fd, int number_of_args, int board_destination)
*       Description:
*       ------------
*	This function supplies the PCI board DSP with two pieces
*	of information: 1) the number of arguments supplied with
*	the next command to be issued, 2) the board destination
*	(PCI, Timing or Utility).
*
*  	NOTE: number_of_args should be mapped in the 16 high bits, and 
*  	board_destination should be mapped in the 16 low bits. Therefore, if
*  	number of arguments is 1, then number_of_args should be 0x00010000.
*	For this reason, when using this function for declare the number_of_args
*	it's advisable to use TDL_ARG_NUM, RDM_ARG_NUM, WRM_ARG_NUM, ... etc.
*
*	Parameters:
*	-----------
*	pci_fd		  File descriptor for the pci device.
*	number_of_args    The number of arguments supplied with
*	                  the next command to be issued.
*	board_destination The board destination (PCI, Timing or Utility).
*
*	Return:
*	-------
*	Returns NO_ERROR if reply is DON, returns ERROR any other case.
*
*	Version: 1.00
*	Author:	 Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    Instead of void, the function will
*					return an int, value,  that is the 
*					result to apply the ioctl command.
*					value is initialized with UNDEFINED
*
*	07/26/00	jmp	1.00	Read reply buffer and check for DON
*					return ERROR or NO_ERROR
*
******************************************************************************/
int set_board_destination(int pci_fd, int number_of_args, int board_destination)
{
	int value;
	int result;
	
	value = 0;

	/*****
	*  set the mask with the number of arguments and the board destination.
	*
	*  NOTE: number_of_args should be mapped in the 16 high bits, and 
	*  board_destination should be mapped in the 16 low bits. Therefore, if
	*  number of arguments is 1, then number_of_args should be 0x00010000.
	******/
	arg_and_board_dest = (number_of_args | board_destination);

	value = arg_and_board_dest;
	ioctl(pci_fd, ASTROPCI_SET_DESTINATION, &value);

	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int set_type(int type)
*
*       Description:
*       ------------
*	This function sets the DSP memory type for the board specified
*	by the <I>set_board_destination()</I> method. The memory
*	spaces are: D, P, R, X, Y.
*
*	Parameter:
*	---------
*	type	One of the following DSP memory types: D, P, R, X, Y.
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    Instead of void, the  function will
*					return an int, result,  that is the 
*					current memory_type.
*
******************************************************************************/
int set_type(int type)
{
	int result;
	result = UNDEFINED;

	memory_type = type;
	result = memory_type;

	return result;
}

/******************************************************************************
*
*       Function:
*       ---------
*		int tdl (int pci_fd, int data)
*
*       Description:
*       ------------
*	Test Data Link (TDL) method. Used to test communications to
*	the PCI, Timing, and Utility boards. A data value is sent to
*	the board specified by the <I>set_board_destination()</I>
*	method. The return value is is checked against the sent value.
*
*	Parameters:
*	---------
*	pci_fd		File descriptor for the pci device.
*	data		Value to be sent to the specified board.
*
*	Returns: 
*	--------
*	Returns the same value that was sent if successful, or ERROR otherwise.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*	If success, reply_value is the value returned for tdl().
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    Name changes from 
*						test_data_link(), to tdl()
*					support for reply_value.
*
*	07/26/00	jmp	1.00	Read reply buffer and check for DON
*					(after set ARG1), if reply is not
*					equals to DON return ERROR
*
******************************************************************************/
int test_data_link(int pci_fd, int data)
{
	int result = NO_ERROR;
				
	/*****
	*  write data to Arg1 register (data is equal to any 24 bit hexa value)
	*  using ioctl().
	******/
	reply_value = data;
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	/*****
	*  write TEST_DATA_LINK coomand to the HCVR register.
	******/
	reply_value = TEST_DATA_LINK;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to data written
	*  to the Arg1 register.
	******/
	result = read_reply (pci_fd, data);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int read_memory (int pci_fd, int address)
*
*       Description:
*       ------------
*	Read Memory (RDM) method. Used to read a DSP memory location
*	from one of the boards (PCI, Timing, or Utility).
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	address		Address (in hex) within DSP of the specified board.
*
*	Returns: 
*	--------
*	Returns the data value located at the specified address, or the flag
*	ERROR.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*	If success, reply_value is the value returned for read_memory().
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    support for reply_value.
*
*	07/26/00	jmp	1.00	Read reply buffer and check for DON
*					(after set ARG1 and after set ARG2), 
*					if reply is not equals to DON returns
*					ERROR
*
******************************************************************************/
int read_memory(int pci_fd, int address)
{
	int result = NO_ERROR;

	/*****
	*  set ARG1 with the memory type to be read.
	******/
	reply_value = memory_type;
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}
	
	/*****
	*  set ARG2 with the address to be read.
	******/
	reply_value = address;
	ioctl(pci_fd, ASTROPCI_SET_ARG2, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}
	
	/*****
	*  does the READ_MEMORY command. 
	******/
	reply_value = READ_MEMORY;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/*****
	* Read the reply.
	******/
	result = read_reply(pci_fd, (int)INT_NULL);
		
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int write_memory (int pci_fd, int address, int data)
*
*       Description:
*	------------
*	Write Memory (WRM) method. Used to write to a DSP memory location
*	on one of the boards (PCI, Timing, or Utility). Writes the specified
*	data value located at address.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	address		Address (in hex) within DSP of the specified board.
*	data		Value to be written to the specified board.
*
*	Returns: 
*	--------
*	Returns the ascii sequence DON if successfull, or the flag ERROR if not.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*	If success, reply_value is equal to DON.
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    support for reply_value.
*
*	07/26/00	jmp	1.00	Read reply buffer and check for DON
*					(after set ARG1, after set ARG2, and
*					after set ARG3). 
*					if reply is not equals to DON returns
*					ERROR
*
******************************************************************************/	
int write_memory(int pci_fd, int address, int data)
{
	int result = NO_ERROR;

	/*****
	*  set ARG1 with the memory type to be write.
	******/
	reply_value = memory_type;
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}
		
	/*****
	*  set ARG2 with the address to be write.
	******/
	reply_value = address;
	ioctl(pci_fd, ASTROPCI_SET_ARG2, &reply_value);

	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}
		
	/*****
	*  set ARG3 with the data to be write.
	******/
	reply_value = data;
	ioctl(pci_fd, ASTROPCI_SET_ARG3, &reply_value);

	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}
		
	/*****
	*  does the WRITE_MEMORY command. 
	******/
	reply_value = WRITE_MEMORY;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
		
	/*****
	*  Read the reply 
	******/
	result = read_reply(pci_fd, DON);
			
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}
	
/******************************************************************************
*
*       Function:
*       ---------
*		int power_on(int pci_fd)
*
*       Description:
*	------------
*	Power on (PON) method. Used to turn the controller power on.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    support for reply_value.
*
******************************************************************************/	
int power_on(int pci_fd)
{
	int result = NO_ERROR;

	reply_value = POWER_ON;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	result = read_reply (pci_fd, DON);
		
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int power_off(int pci_fd)
*
*       Description:
*	------------
*	Power off method. Used to turn the controller power off.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/2000 	jmp     1.00    Initial
*
*
******************************************************************************/	
int power_off(int pci_fd)
{
	int result = NO_ERROR;

	reply_value = POWER_OFF;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	result = read_reply (pci_fd, (int)INT_NULL);
		
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}


/******************************************************************************
*
*       Function:
*       ---------
*		int start_exposure(int pci_fd)
*
*       Description:
*	------------
*	Start exposure (SEX) method. Used to start an image exposure.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*	Returns ascii sequence "DON" if successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/2000 	jmp     1.00    Initial
*
******************************************************************************/
int start_exposure(int pci_fd)
{
	int result = NO_ERROR;
				
	reply_value = START_EXPOSURE;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	result = read_reply (pci_fd, DON);
		
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int read_image(int pci_fd, int byte_count, 
*						int exposure_time, int mem_fd)
*       Description:
*	------------
*	Read image (RDI) method. Used to readout an image from the controller
*	into the user application.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	byte_count	The number of bytes to readout.
*	exposure_time	The image's exposure time.
*	mem_fd		Pointer to the application image buffer.
*
*	Returns: 
*	--------
*	Returns ascii sequence "DON" if successfull.
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/2000 	jmp     1.00    Initial
*
*
******************************************************************************/	
int read_image(int pci_fd, int byte_count, int exposure_time, int mem_fd)
{
	int result = NO_ERROR;
        void *addrPtr;		/* local pointer to the user image buffer  */

	/******** 
        * To handle when the exposure time is bigger than EXP_TIME_LOW_LIMIT
	* (see DSPCommand.h for this value), the  READ_IMAGE command is sent
	* to the controller.
	********/
	if (exposure_time >= EXP_TIME_LOW_LIMIT)
	{
		reply_value = READ_IMAGE;
		ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	}
	
	/******** 
        * Obtain a pointer to the user's image buffer.
	********/
        addrPtr = (unsigned char *)mem_fd;

	/******** 
        *  Call the device driver to "read". 
	*  Transfer the image from PCI into the image buffer.
     	*  The read() function attempts to read nbyte  bytes  from  the
	*  file  associated with the open file descriptor, fildes, into
	*  the buffer pointed to by buf.
	*	read() returns a ssize_t value, the size of bytes read.
	********/
        result = read(pci_fd, addrPtr, byte_count);

	/******** 
        * Check DMA timeout. 
	********/
        if (result == EIO) 
	{
		sprintf(str_reply_value, 
			"DMA timeout waiting for image data (errno = %d) %s\n",
 			errno, strerror(errno));
        }
	/******** 
        * Check DMA timeout. 
	********/
        else if (result == EFAULT) 
	{
		sprintf(str_reply_value, 
                	"Image Readout EFAULT");
	}
	/******** 
        * Check abort readout. 
	********/
        else if (result == EINTR) 
	{
		sprintf(str_reply_value, 
                	"Image Readout Successfully Aborted");
        }
	/******** 
        * Check image buffer state corruption.
	********/
        else if (result == ECHILD) 
	{
		sprintf(str_reply_value, 
			"Image buffer in a corrupt state (errno = %d) \n%s", 
			errno, strerror(errno));
        }

	/******** 
        * Check for miscellanous errors.
	********/
        else if (result == -1) 
	{
                if (errno != 0)
		{
                	sprintf(str_reply_value, 
				"Error While Reading Image (errno = %d) \n%s",
				errno, strerror(errno));
		}
        }

	/******** 
        * Last Check for miscellanous errors.
	********/
        if (result <0) 
	{
		reply_value = ERR;
		sprintf (str_reply_value, 
		   "Error when trying to read from pci into memory buffer.");
		result = ERROR;
		return result;

	} else {
		/* read_image (pci_fd, mem_fd, byte_count); */
	
		/* Read the reply. */
		result = read_reply(pci_fd, DON);
	}

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
	/********
	return result;
	********/

}

/******************************************************************************
*
*	Dimensions method. Used to set the image dimensions.
*
*	PARAMETERS:
*	----------
*	ccd_rows	The number of image rows.
*	ccd_cols	The number of image columns.
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/
int dimensions(int pci_fd, int ccd_rows, int ccd_columns)
{
	int result = NO_ERROR;
	int value;
		
	value = ccd_columns;
	ioctl(pci_fd, ASTROPCI_SET_NCOLS, &value);
	
	/* Read the reply. */
	result = read_reply (pci_fd, DON);

	if ( (result = check_error_reply_value(result)) == ERROR)
	{
		return result;
	}

	value = ccd_rows;
	ioctl(pci_fd, ASTROPCI_SET_NROWS, &value);

	/* Read the reply. */
	result = read_reply(pci_fd, DON);
		
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Read exposure time function. Used to read the current elapsed exposure
*	time.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	Returns the current elapsed exposure time.
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/		
int read_expose_time(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = READ_EXPOSURE_TIME;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, (int)INT_NULL);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Set exposure time method. Used to set the exposure time to a new value.
*
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	exposure_time	the value for to set the exposure time.
*
*	RETURNS:	Returns the current elapsed exposure time.
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/		
int set_exposure_time (int pci_fd, int exposure_time)
{
	int result = NO_ERROR;
		
	reply_value = exposure_time;
	ioctl(pci_fd, ASTROPCI_SET_EXPTIME, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Abort exposure function. Used to abort the current image exposure.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/		
int abort_exposure(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = ABORT_EXPOSURE;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	
	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Pause exposure function. Used to pause the current image exposure.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/		
int pause_exposure(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = PAUSE_EXPOSURE;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	
	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Resume exposure function. Used to resume a paused image exposure.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/		
int resume_exposure(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = RESUME_EXPOSURE;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	
	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Abort exposure readout function. Used to abort the current exposure 
*	readout.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/		
int abort_readout(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = ABORT_READOUT;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	
	/* Read the reply. */
	result = read_reply(pci_fd, (int)INT_NULL);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Reset controller function. Used to reset the controller.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/
int reset_controller(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = RESET_CONTROLLER;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, SYR);

	if (reply_value == SYR)
	{
		result = NO_ERROR;
		sprintf(str_reply_value,"Reset done. Command was successfull.");
	}

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Open shutter function. Used to open the camera shutter.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns:	DON or ERROR
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    
*					support for reply_value.
*					reply_value=OPEN_SHUTTER instead of
*					reply_value=_OPEN_SHUTTER
*					ioctl( , ASTROPCI_SET_HCVR, ) instead of
*					ioctl( , ASTROPCI_SET_UTIL_OPTIONS, )
*
******************************************************************************/
int open_shutter(int pci_fd)
{
	int result = NO_ERROR;
		
	reply_value = OPEN_SHUTTER;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Close shutter function. Used to close the camera shutter.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns:	DON or ERROR
*
*	Version: 1.00
*	Author:  Scott Streit
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       11/22/99        sds     1.00    Initial
*
*       06/20/00        jmp     1.00    
*					support for reply_value.
*					reply_value=CLOSE_SHUTTER instead of
*					reply_value=_CLOSE_SHUTTER
*					ioctl( , ASTROPCI_SET_HCVR, ) instead of
*					ioctl( , ASTROPCI_SET_UTIL_OPTIONS, )
*
******************************************************************************/
int close_shutter(int pci_fd)
{
	int result = NO_ERROR;
	
	reply_value = CLOSE_SHUTTER;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*	This method specifies whether or not to open the shutter during an
*	exposure. The command goes to the Utility board. A "1" is sent to the
*	Constant ASTROPCI_SET_UTIL_OPTIONS register for "open shutter"
*	and a "0" is sent for "close shutter".
*
*	NOTE:
*	-----
*	Because this function forces shutter position and it is not advisable
*	to use, unless the shutter position is know in advance. For open or
*	close shutter use set_shutter_position() instead.
*
*	PARAMETER:
*	----------
*	pci_fd		Pointer to the opened PCI device.
*	open_shutter 	If true, open the shutter during an exposure. If
*		       	false, do not open the shutter during an exposure.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/
int shutter_position(int pci_fd, int open_shutter)
{
	int result = NO_ERROR;
		
	/* Ask for the current status of the controller. */
	reply_value = READ_CONTROLLER_STATUS;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, (int)INT_NULL);

	if (open_shutter == TRUE)
	{
		/* Open the shutter via the utility board options. */
		reply_value = reply_value | _OPEN_SHUTTER_;
		ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl
		******/
		result = read_reply (pci_fd, (int)INT_NULL);

		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}
	else if (open_shutter != TRUE)
	{
		/* Close the shutter via the utility board options. */
		reply_value = reply_value | _OPEN_SHUTTER_;
		ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl
		******/
		result = read_reply (pci_fd, (int)INT_NULL);

		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}

	/* writes to the controller. */
	reply_value = WRITE_CONTROLLER_STATUS;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
		
	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Idle method. Used to put the controller into an idle state or take
*	the controller out of the idle state.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	mode		True (1) to put the controller into an idle state.
*		    	False (0) to take the controller out of the idle state.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/	
int idle(int pci_fd, int mode)
{
	int result = NO_ERROR;
		
	/********
	* Resume idle mode.
	********/
	if (mode) 
	{
		reply_value = RESUME_IDLE_MODE;
		ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	}

	/********
	* Stop idle mode. 
	********/
	else 
	{
		reply_value = STOP_IDLE_MODE;
		ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
	}
	
	/* Read the reply. */
	result = read_reply(pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*	Gain function. Used to set the controller readout gain and A/D
*	conversion speed.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	gain 		The readout gain.
*	speed 		The A/D conversion speed. Can be "fast" or "slow".
*
*	Returns:	if error, returns ERROR, otherwise returns the value
*			of reply_value that should be equals to DON.
*			
*	Other Effects:	The str_reply_value and reply_value are loaded with
*			the result of send the DSP command, either success
*			or fail. 
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/

int set_gain(int pci_fd, int gain, int speed)
{
	int result = NO_ERROR;

	/********
	*  set the board destination.
	********/
	set_board_destination (pci_fd, 0x20000, TIM_ID);

	/********
	*  write desired gain value to the arg1.
	********/
	reply_value = gain;
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	/********
	*  write desired speed value to the arg2.
	********/
	reply_value = speed;
	ioctl(pci_fd, ASTROPCI_SET_ARG2, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	/********
	*  send manual command: write ascii string SGN to the command register.
	********/
	reply_value = SGN;
	ioctl(pci_fd, ASTROPCI_SET_CMDR, &reply_value);

	/********
	* Read the reply.
	********/
	ioctl (pci_fd, ASTROPCI_GET_CMDR, &reply_value);
	
	/********
	* reply should be equal to DON.
	********/
	if (reply_value != DON)
	{
		result = ERROR;
		switch (reply_value)
		{
			case -1:
				sprintf (str_reply_value, 
				"Error: Set Gain & Speed. Reply buffer empty.");
				break;
			case EMTY :
				sprintf (str_reply_value, 
				"Error: Set Gain & Speed. Reply buffer empty.");
				break;
			case TOUT :
				sprintf (str_reply_value, 
				"Error: Set Gain & Speed. Reply timeout.");
				break;
			case SYR :
				sprintf (str_reply_value, 
				"Error: Set Gain & Speed. External reset.");
				break;
			default:
				sprintf (str_reply_value, 
				"Error: Set Gain & Speed. (reply=0x%06X)", 
								reply_value);
				break;
		}

	} else {
		result = NO_ERROR;
		sprintf (str_reply_value, "DON read. Command was successfull.");
	}

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}
	

	
/*****************************************************************************
*
*	Load application method. Used to load a timing or utility board
*	application.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	app_number	The number of the application to load.
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/
int load_application(int pci_fd, int board_id, int app_number)
{
	int result = NO_ERROR;

	if (board_id==TIM_ID)
	{
		set_board_destination (pci_fd, TIM_APP_ARG_NUM, board_id);
	}
	else if (board_id==UTIL_ID)
	{
		set_board_destination (pci_fd, UTIL_APP_ARG_NUM, board_id);
	}
	else
	{
		return ERROR;
	}

	reply_value = app_number;
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	reply_value = LOAD_APPLICATION;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/* Read the reply. */
	result = read_reply(pci_fd, DON);
	
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Set temperature method. Used to set the detectors target temperature.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	temperature	The detector temperature (in Celsius).
*
*	RETURNS:	None
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/	
int set_temperature(int pci_fd,  double temperature)
{
	int result = NO_ERROR;

	reply_value = calculate_adu (temperature);
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	reply_value = SET_ARRAY_TEMPERATURE;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply(pci_fd, DON);
	
	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Get temperature method. Used to obtain the current array temperature
*	from the controller.
*
*	PARAMETER:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	The current array temperature.
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/	
double get_temperature(int pci_fd)
{
	int reply_value;		/*  DSP reply value	  */
	int adu = 0;
	double temperature = KELVIN_TO_CELSIUS; 

	/********
	* Get the temperature representation in ADU's
	********/
	reply_value = READ_ARRAY_TEMPERATURE;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);

	adu = reply_value;
	adu = adu & 0x00000FFF;

	temperature= calculate_temperature((int)adu);
			
	/********
	* return result.
	********/
	return temperature;
}

/******************************************************************************
* FUNCTION:	read_reply
*
* INPUT:	expected_value - A value to compare against the reply value. If
* 				 no compare value is needed, use 'INT_NULL'.
* 		pci_fd - A reference to the device driver that was obtained
* 			 by using the "open" command.
* 			 
* OUTPUT:	Returns the controller reply value.
*
* DESCRIPTION:	This function retrieves the current controller reply. A compare
* 		value may be supplied, which will be used to check the reply
* 		value automatically. A timeout will occur after MAX_WAIT_TIME
* 		seconds.
*
* Author:	Scott Streit
* Date:		01/10/2000
*
*   Revision History:
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       01/10/2000	sds	1.00	Initial
*
*       06/20/00        jmp     1.00    
*					error messages redirected to stderr. 
*					But also are commented.
*					ASTROPCI_CLEAR_REPLY changes to
*					ASTROPCI_FLUSH_REPLY_BUFFER
*
*					Add the section Check for a "hardware 
*					button" reset 
*
*					All check sections are completed by
*					returning a value, ususally the get
*					reply_value. Always,  before return 
*					flushes the reply buffer.
*
******************************************************************************/
int read_reply (int pci_fd, int expected_value)
{
	int result= NO_ERROR;
	int tmp_reply_value = 0;
	time_t start_time = time(NULL);
	double timer = 0;

	ioctl (pci_fd, ASTROPCI_GET_REPLY, &tmp_reply_value);
	
	/****** 
	* Wait for a reply to become available. 
	******/
	while (tmp_reply_value == -1) {
		ioctl(pci_fd, ASTROPCI_GET_REPLY, &tmp_reply_value);
		timer = difftime(time(NULL), start_time);
		if (timer > (double)MAX_WAIT_TIME) 
		{
			break;
		}
	}

	/****** 
	* If there is ERR, set reply value to ERR. 
	******/
	if (tmp_reply_value == ERR) {
		
		result = ERROR;
		reply_value = ERR;
		sprintf (str_reply_value, "Error: Reply buffer error");

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);

		return result;
	}

	else
	
	/****** 
	* If there is no buffer value, set reply value to ERR. 
	******/
	if (tmp_reply_value == -1) {
		
		result = ERROR;
		reply_value = ERR;
		sprintf (str_reply_value, "Error: Reply buffer empty");

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);

		return result;
	}

	else
	
	/****** 
 	* Check for a timeout to prevent a hangup.
	******/
	if (timer > MAX_WAIT_TIME) {
		
		result = ERROR;
		reply_value = TOUT;
		sprintf (str_reply_value, "Error: Reply buffer timeout.");

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);

		return result;
	}

	else

	/****** 
	*  Check for a "hardware button" reset.
	******/
	if ( (tmp_reply_value == RST) || (tmp_reply_value == SYR) ) {

		result = ERROR;
		reply_value = tmp_reply_value;
		sprintf (str_reply_value, 
			"The contoller has experienced an external reset. ");
		sprintf (str_reply_value, 
			 "%sPlease re-apply the contoller setup.", 
			 str_reply_value);

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);

		return result;
	}
	
	else

	/****** 
	*  Compare the reply value against the user's expected value. 
	******/
	if ((expected_value!=(int)INT_NULL)&&(tmp_reply_value!=expected_value)){

		result = ERROR;
		reply_value = tmp_reply_value;
		sprintf (str_reply_value,
			"Error: Reply value not equal to expected value\n");
		sprintf (str_reply_value, 
			 "%s\tActual = 0x%08X\n",str_reply_value, reply_value);
		sprintf (str_reply_value, 
			 "%s\tExpected = 0x%08X", str_reply_value, 
			 expected_value);

		if (reply_value == TOUT) {
			result = ERROR;
			reply_value = TOUT;
			sprintf (str_reply_value, 
					"Error: Reply buffer timeout.");
		}
		else
		if (reply_value == ERR) {
			result = ERROR;
			reply_value = ERR;
			sprintf (str_reply_value,"Error: Reply buffer empty");
		}

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);

		return result;
	}

	else 

	/****** 
	* check for successfull DON (Done)
	******/
	if (tmp_reply_value == DON) {
		result = NO_ERROR;
		reply_value = DON;
		sprintf (str_reply_value,
				"DON read. Command was successfull.");

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);

		return result;
	}

	/****** 
	* Case were command is successfull and the reply_value has a value.
	******/
	else {

		result = NO_ERROR;
		reply_value = tmp_reply_value;
		sprintf (str_reply_value, "Command was successfull.");

		/* Re-initialize the reply buffer. */
		tmp_reply_value = 0;
    		ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &tmp_reply_value);
		
	}

	return result;   
}


/******************************************************************************
*	Utility function that checks the value of the parameter error. If the
*	parameter error is equal to ERROR, the function returns ERROR, if not
*	the function returns the current value of the variable reply_value.
*
*	PARAMETER:
*	---------
*	error		an int value to be checked if is equal to ERROR.
*
*	RETURNS:	if ERROR is equal to ERROR, returns ERROR, otherwise
*			returns the current value of reply_value.
*
*	Version: 1.00
*	Author:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/	
int check_error_reply_value (int error)
{
	if (error == ERROR ) 
	{
		return error;
	}

	return reply_value;
}


/*****************************************************************************
*	Flush PCI board reply buffer. This method flushes the PCI board reply
*	buffer.
*
*	Parameters:
*	-----------
*	pci_fd		Pointer to the opened PCI device.
*
*	RETURNS:	The content of reply buffer after flush, that should be
*			0x00000000
*
*	VERSION: 1.00
*	AUTHOR:  Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
*****************************************************************************/
int flush_reply_buffer(int pci_fd)
{
	int value;

	value = 0;	
	ioctl(pci_fd, ASTROPCI_FLUSH_REPLY_BUFFER, &value);

	return value;
}


/*****************************************************************************
*
*       Function:
*       ---------
*		int download_mode_set_argument1(int pci_fd, int value)
*
*       Description:
*	------------
*		Set the argument1 register with the desired value.
*
*	NOTE: this function should be only used to set the argument1 register
*	when the controller is in download mode --see set_pci_to_download_mode()
*	Because the controller in the "pci file download" mode, this is a case
*	where after set the argument1 register, the reply buffer shouldn't be
*	read.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	value		A value for to set the argument1 register.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*****************************************************************************/
int download_mode_set_argument1(int pci_fd, int value)
{
	int result = NO_ERROR;
	int reply = 0;

	/********
	* set the PCI ARG1 register with the value. 
	********/
	reply = value;
	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply);
		
	/********
	* check for error in reply, update reply_value and str_reply_value.  
	********/
	if (reply == value)
	{
		result = reply;
		sprintf(str_reply_value, "ARG1 set to value 0x%08X.", value); 
		sprintf(str_reply_value, "%s Command was successfull", 
				str_reply_value); 
	} else {
		result = ERROR;
		sprintf(str_reply_value, "Cannot set ARG1 to value 0x%08X.",
				value); 
	}
	reply_value = reply;

	/********
	* return result.
	********/
	return result;
}


/*****************************************************************************
*
*       Function:
*       ---------
*		int set_pci_to_download_mode(int pci_fd)
*
*       Description:
*	------------
*		Set the PCI board to download mode. The download mode is 
*	inteded to download an application file into the pci board.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*****************************************************************************/
int set_pci_to_download_mode(int pci_fd)
{
	int result = NO_ERROR;
	int reply;

	reply = 0;

        /**********
        * Set the PCI board to slave mode.
        **********/
        ioctl(pci_fd, ASTROPCI_GET_HCTR, &reply);

        /**********
        * Clear the HTF bits.
        **********/
        reply = reply & HTF_CLEAR_MASK;

        /**********
        * Clear bit 3.
        **********/
        reply = reply & BIT3_CLEAR_MASK;

        /**********
        * Set the HTF bits.
        **********/
        reply = reply | HTF_MASK;
        ioctl(pci_fd, ASTROPCI_SET_HCTR, &reply);

        /**********
        * Inform the DSP that new pci boot code will be downloaded.
        **********/
        reply = PCI_DOWNLOAD;
        ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply);

        /**********
        * Set the magic value that says this is a PCI download.
        **********/
        reply = 0x00555AAA;
        ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply);
		
        /**********
        * return the result
        **********/
	result= reply;
	sprintf(str_reply_value, "PCI set to download mode."); 

	return result;
}

/*****************************************************************************
*
*       Function:
*       ---------
*		int set_pci_to_transfer_mode(int pci_fd)
*
*       Description:
*	------------
*		Set the PCI board to transfer mode. Normal work mode in order
*	to do the readouts.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*****************************************************************************/
int set_pci_to_transfer_mode(int pci_fd)
{
	int result = NO_ERROR;
	int reply;

        /**********
        * Set the PCI board data transfer format (Set HTF bits to 00).
        **********/
        ioctl(pci_fd, ASTROPCI_GET_HCTR, &reply);
        reply = reply & HTF_CLEAR_MASK;
        ioctl(pci_fd, ASTROPCI_SET_HCTR, &reply);

        /**********
        * Set bit 3.
        **********/
        reply = reply | BIT3_SET_MASK;
        ioctl(pci_fd, ASTROPCI_SET_HCTR, &reply);

        /**********
        * return the result
        **********/
	result= reply;
	sprintf(str_reply_value, "PCI set to transfer mode."); 

	return result;
}

/*****************************************************************************
*
*       Function:
*       ---------
*		int get_hctr_register (int pci_fd)
*
*       Description:
*	------------
*	Get the HCTR register.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*
*	Returns: 
*	--------
*	If success, returns the value of the HCTR register.
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*****************************************************************************/
int get_hctr_register (int pci_fd)
{
	int result= NO_ERROR;

	/********
	* first of all, send some data to test the data link.
	********/
	set_board_destination (pci_fd, TDL_ARG_NUM, PCI_ID);

	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	reply_value = TEST_DATA_LINK;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
		
	/********
	* get the HCTR register
	********/
	ioctl (pci_fd, ASTROPCI_GET_HCTR, &reply_value);


	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}


/*****************************************************************************
*
*       Function:
*       ---------
*		int set_hctr_register (int pci_fd, int value)
*
*       Description:
*	------------
*	Set the HCTR register with the value that is passed as a parameter.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	value		value for to set the HCTR register.
*
*	Returns: 
*	--------
*	Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 07/10/2000
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       07/10/2000 	jmp     1.00    Initial
*
*
*****************************************************************************/
int set_hctr_register (int pci_fd, int value)
{
	int result= NO_ERROR;

	/********
	* first of all, send some data to test the data link.
	********/
	set_board_destination (pci_fd, TDL_ARG_NUM, PCI_ID);

	ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
	/*****
	*  Read the reply buffer using ioctl, should be equal to DON
	******/
	result = read_reply (pci_fd, DON);

	/********
	* check for error in reply_value. 
	********/
	if (check_error_reply_value(result)==ERROR) 
	{	
		return ERROR;
	}

	reply_value = TEST_DATA_LINK;
	ioctl(pci_fd, ASTROPCI_SET_HCVR, &reply_value);
		

	/********
	* set the HCTR register.
	********/
	reply_value = value;
	ioctl (pci_fd, ASTROPCI_SET_HCTR, &reply_value);

	/********
	* read the HCTR register in order to return it back.
	reply_value = get_hctr_register (pci_fd);
	********/
	/********
	* get the HCTR register
	********/
	ioctl (pci_fd, ASTROPCI_GET_HCTR, &reply_value);

	if (reply_value != value)
	{
		result = ERROR;
	}

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/*****************************************************************************
*
*       Function:
*       ---------
*		int set_hcvr(int pci_fd, int command, int expected_reply)
*
*       Description:
*	------------
*	Send a command to the PCI DSP via the vector register. The reply
*	returned from the PCI DSP is checked against the specified expected
*	reply value.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	command 	The command to be sent to the PCI DSP.
*	expected_reply	The expected reply value.
*
*	Returns: 
*	--------
*		Returns ERROR if not successfull.
*
*	Other effects:
*	--------------
*		reply_value and str_reply_value are updated.
*
*	Version: 1.00
*	Author:  Scott Streit
*	Date: 08/07/2000
*
*   Revision History:
*   -----------------
*       Date            Who   Version    Description
*   --------------------------------------------------------------------------
*       08/07/2000 	sds     1.00    Initial
*
*
*****************************************************************************/
int set_hcvr(int pci_fd, int command, int expected_reply)
{
	int result = NO_ERROR;
		
	ioctl(pci_fd, ASTROPCI_SET_HCVR, command);
		
	/* Read the reply. */
	result = read_reply (pci_fd, expected_reply);

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*	Function:
*	---------
*	     int send_manual_command(int pci_fd, int board_id, int command_id, 
*			int arg_1, int arg_2, int arg_3, int arg_4, int arg_5)
*
*	Description:
*	------------
*	This function sends a manual command to the PCI driver. 
*
*	Parameters:
*	---------
*	pci_fd		pointer to the opened PCI device.
*	board_id	destination board of the desired manual command.
*	command_id	ascii string corresponding to the desired command.
*	arg_1		argument1 for the manual command. Could be ARG_NULL.
*	arg_2		argument2 for the manual command. Could be ARG_NULL.
*	arg_3		argument3 for the manual command. Could be ARG_NULL.
*	arg_4		argument4 for the manual command. Could be ARG_NULL.
*	arg_5		argument5 for the manual command. Could be ARG_NULL.
*
*	Returns:
*	--------
*	if error, returns ERROR, otherwise returns the value
*	of reply_value that should be equals to DON.
*			
*	Other Effects:
*	--------------
*	The str_reply_value and reply_value are loaded with
*	the result of send the DSP command, either success
*	or fail. 
*
*	Version: 1.00
*	Author: Jose Maria Panero Ciprian
*	Date:	07/10/2000
*
******************************************************************************/

int send_manual_command(int pci_fd, int board_id, int command_id, 
			int arg_1, int arg_2, int arg_3, int arg_4, int arg_5)
{
	int result = NO_ERROR;
	int num_arg = 0;
	int reply = 0;

	/********
	*  try to figure out the number of arguments.
	********/
	if ( arg_1 == ARG_NULL )
	{
		num_arg = 0;
	} else if ( arg_2 == ARG_NULL )
	{
		num_arg = 1;
	} else if ( arg_3 == ARG_NULL )
	{
		num_arg = 2;
	} else if ( arg_4 == ARG_NULL )
	{
		num_arg = 3;
	} else if ( arg_5 == ARG_NULL )
	{
		num_arg = 4;
	} else {
		num_arg = 5;
	}

	/********
	*  set the board destination.
	********/
	set_board_destination (pci_fd, (num_arg << 16), board_id);

	/********
	*  write desired value to the arg1.
	********/
	if (num_arg >= 1)
	{
		reply_value = arg_1;
		ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl, should be equal to DON
		******/
		result = read_reply (pci_fd, DON);
	
		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}

	/********
	*  write desired value to the arg2.
	********/
	if (num_arg >= 2)
	{
		reply_value = arg_2;
		ioctl(pci_fd, ASTROPCI_SET_ARG2, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl, should be equal to DON
		******/
		result = read_reply (pci_fd, DON);
	
		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}

	/********
	*  write desired value to the arg3.
	********/
	if (num_arg >= 3)
	{
		reply_value = arg_3;
		ioctl(pci_fd, ASTROPCI_SET_ARG3, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl, should be equal to DON
		******/
		result = read_reply (pci_fd, DON);
	
		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}

	/********
	*  write desired value to the arg4.
	********/
	if (num_arg >= 4)
	{
		reply_value = arg_4;
		ioctl(pci_fd, ASTROPCI_SET_ARG4, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl, should be equal to DON
		******/
		result = read_reply (pci_fd, DON);
	
		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}

	/********
	*  write desired value to the arg5.
	********/
	if (num_arg == 5)
	{
		reply_value = arg_5;
		ioctl(pci_fd, ASTROPCI_SET_ARG5, &reply_value);
		
		/*****
		*  Read the reply buffer using ioctl, should be equal to DON
		******/
		result = read_reply (pci_fd, DON);
	
		/********
		* check for error in reply_value. 
		********/
		if (check_error_reply_value(result)==ERROR) 
		{	
			return ERROR;
		}
	}

	/********
	*  send manual command: issues command to the command register.
	********/
	reply = command_id;
	ioctl(pci_fd, ASTROPCI_SET_CMDR, &reply);

	/********
	* Read the reply.
	ioctl (pci_fd, ASTROPCI_GET_CMDR, &reply_value);
	********/
	result = read_reply (pci_fd, (int)INT_NULL);
	
	/********
	* reply. 
	********/
	switch (reply_value)
	{
		case -1:
			sprintf (str_reply_value, 
				"Error: Manual Command. Reply buffer empty.");
			result = ERROR;
			break;
		case ERR :
			sprintf (str_reply_value, 
				"Manual Command: Reply buffer error.");
			result = ERROR;
			break;
		case EMTY :
			sprintf (str_reply_value, 
				"Error: Manual Command. Reply buffer empty.");
			result = ERROR;
			break;
		case TOUT :
			sprintf (str_reply_value, 
				"Error: Manual Command. Reply timeout.");
			result = ERROR;
			break;
		case SYR :
			sprintf (str_reply_value, 
				"Manual Command. External reset.");
			result = NO_ERROR;
			break;
		default:
			sprintf (str_reply_value, "Command was successfull.");
			result = NO_ERROR;
			break;
	}

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);
}

/******************************************************************************
*
*       Function:
*       ---------
*		int set_shutter_position(int pci_fd, int do_open_shutter)
*
*       Description:
*	------------
*	Set shutter position. If do_open_shutter is TRUE, open the shutter
*	during an exposure. If do_open_shutter is FALSE, close the shutter
*	during an exposure.
*
*	Parameters:
*	---------
*	pci_fd		Pointer to the opened PCI device.
*	do_open_shutter	If TRUE opens the shutter, if FALSE closes the shutter.
*
*	Returns: 
*	--------
*	Returns the ascii sequence DON if successfull, or the flag ERROR if not.
*
*	Other effects:
*	--------------
*	reply_value and str_reply_value are updated.
*	If success, reply_value is equal to DON.
*
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/03/2000
*
*****************************************************************************/
int set_shutter_position(int pci_fd, int do_open_shutter)
{
	int result = NO_ERROR;
	int current_status = 0;

	/*************
	*  Set the board destination to Timing.
	*************/
	result= set_board_destination(pci_fd, RDM_ARG_NUM, TIM_ID);

	if (result==ERROR)
	{
		return result;
	}

	/*************
	*  Set the memory type to X.
	*************/
	set_type(X);

	/*************
	*  Ask for the current controller status from X:0.
	*************/
	current_status = read_memory(pci_fd, 0);

	if (current_status==ERROR)
	{
		return result;
	}

	/*************
	*  Set the board destination to Timing.
	*************/
	result= set_board_destination(pci_fd, WRM_ARG_NUM, TIM_ID);

	if (result==ERROR)
	{
		return result;
	}

	/*************
	*  Set the memory type to X.
	*************/
	set_type(X);

	result= NO_ERROR;

	/*************
	*  Open or close the shutter after the value of "do_open_shutter".
	*************/
	if (do_open_shutter) 
	{
		/*************
		*  Open the shutter.
		*************/
		result= write_memory(
			pci_fd, 0, (current_status | _OPEN_SHUTTER_));

	}
	else if (!do_open_shutter) 
	{
		/*************
		*  Close the shutter.
		*************/
		result= write_memory(
			pci_fd, 0, (current_status & _CLOSE_SHUTTER_));
	}

	return result;
}

/******************************************************************************
*
*	Function:
*	---------
*		int arg(int pci_fd, int arg_number, int arg_value)
*
*	Set one of the PCI DSP argument registers to the specified value. The
*	argument register should be 1, 2, 3, 4 or 5. 
*
*	Parameters:
*	---------
*	arg_number	an int value that indicates the argument register
*			to be set. Should be 1, 2, 3, 4, or 5.
*	arg_value	an int value to which the argument register should be
*			set.
*
*	Returns:	
*	--------	Returns NO_ERROR, if the command is successfull, and
*			ERROR otherwise.
*			If the arg_number is not 1, 2, 3, 4, or 5 returns 
*			ERROR.
*
*	Other effects:	
*	--------------	reply_value and str_reply_value are updated with
*			the result of execute this command.
*
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/17/2000
*
******************************************************************************/	
int arg(int pci_fd, int arg_number, int arg_value)
{

	int result = NO_ERROR;

	switch (arg_number) {

		/*****
		* case: set the argument register #1
		******/
		case 1:
			reply_value = arg_value;
			ioctl(pci_fd, ASTROPCI_SET_ARG1, &reply_value);
			break;

		/*****
		* case: set the argument register #2
		******/
		case 2:
			reply_value = arg_value;
			ioctl(pci_fd, ASTROPCI_SET_ARG2, &reply_value);
			break;

		/*****
		* case: set the argument register #3
		******/
		case 3:
			reply_value = arg_value;
			ioctl(pci_fd, ASTROPCI_SET_ARG3, &reply_value);
			break;

		/*****
		* case: set the argument register #4
		******/
		case 4:
			reply_value = arg_value;
			ioctl(pci_fd, ASTROPCI_SET_ARG4, &reply_value);
			break;

		/*****
		* case: set the argument register #5
		******/
		case 5:
			reply_value = arg_value;
			ioctl(pci_fd, ASTROPCI_SET_ARG5, &reply_value);
			break;

		/*****
		* default case means the argument register specified is
		* not 1, 2, 3, 4, or 5. In this case update the reply_value
		* and str_reply_value to error, and return ERROR.
		******/
		default:
			sprintf(str_reply_value, 
			   "Error: can't set argument register #%d. %s.", 
			   "Argument register # should be 1, 2, 3, 4, or 5");
			reply_value = ERR;
			return ERROR;
			break;
	}

	/*****
	*  At this point the argument register number was 1, 2, 3, 4, or 5. The
	*  PCI DSP argument register was set with the specified value, so
	*  the reply value should be read.
	******/
	result = read_reply (pci_fd, DON);
	
	/********
	* check for error in reply_value. 
	********/
	return check_error_reply_value(result);
}

/******************************************************************************
*
*	Function:
*	---------
*		int cmdr(int pci_fd, int command)
*
*	Issues the specified manual command to the PCI DSP.
*
*	Parameters:
*	---------
*	command		an int that is the hexadecimal value of the command
*			to be issued using the manual command feature in the
*			PCI DSP.
*
*	Returns:	
*	--------	Returns NO_ERROR, if the command is successfull, and
*			ERROR otherwise.
*
*	Other effects:	
*	--------------	reply_value and str_reply_value are updated with
*			the result of execute this command.
*
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/17/2000
*
******************************************************************************/	
int cmdr(int pci_fd, int command)
{
		
	int result = NO_ERROR;
	int reply;

	/********
	*  issues command to the command register.
	********/
	reply = command;
	ioctl(pci_fd, ASTROPCI_SET_CMDR, &reply);

	/********
	* Read the reply.
	********/
	result = read_reply (pci_fd, (int)INT_NULL);
	
	/********
	* check reply. 
	********/
	switch (reply_value)
	{
		case -1:
			sprintf (str_reply_value, 
				"Error: Manual Command. Reply buffer empty.");
			result = ERROR;
			break;
		case ERR :
			sprintf (str_reply_value, 
				"Manual Command: Reply buffer error.");
			result = ERROR;
			break;
		case EMTY :
			sprintf (str_reply_value, 
				"Error: Manual Command. Reply buffer empty.");
			result = ERROR;
			break;
		case TOUT :
			sprintf (str_reply_value, 
				"Error: Manual Command. Reply timeout.");
			result = ERROR;
			break;
		case SYR :
			sprintf (str_reply_value, 
				"Manual Command. External reset.");
			result = NO_ERROR;
			break;
		default:
			sprintf (str_reply_value, "Command was successfull.");
			result = NO_ERROR;
			break;
	}

	/********
	* check for error in reply_value. 
	********/
	return  check_error_reply_value(result);

}

/******************************************************************************
*
*	Function:
*	---------
*		char *to_ascii(int value)
*
*	Converts a 24-bit value into an ASCII string.
*
*	PARAMETER:
*	---------
*	value		24-bit integer to convert to ascii.
*
*	Returns:	
*	--------	The string ascii value of the input integer.
*
*	VERSION: 1.00
*	AUTHOR:  Scott Streit
*
******************************************************************************/	
char *to_ascii(int value)
{
	ascii_value[0] = (char)((value & 0xff0000) >> 16);
	ascii_value[1] = (char)((value & 0x00ff00) >> 8);
	ascii_value[2] = (char)(value & 0x0000ff);
	
	return ascii_value;
}


/******************************************************************************
*
*	Function:
*	---------
*		int to_value(char *text)
*
*	Converts an ASCII string into a 24-bit integer. The string should have
*	three characters length, otherwise ERROR is returned. The function is
*	not lowercase sensitive. The characters in the input string are 
*	converted to uppercase.
*
*	Parameters:
*	---------
*	text		ascii string to be converted to 24-bit value.
*
*	Returns:	
*	--------	The 24-bit value result of convert the input string. 
*			Returns ERROR if the length of the string input is 
*			different than 3 characters.
*
*
*	Version: 1.00
*	Author:  Jose Maria Panero
*	Date: 08/11/2000
*
******************************************************************************/	
int to_value(char *text)
{
	int a, b, c;
	int result= 0;
	int len= strlen(text);
	int i;

	if (len != 3) return ERROR;

        /********
	*  Get the uppercase value of the char in the input string
        ********/
	a= (int)(toupper(text[0]));
	b= (int)(toupper(text[1]));
	c= (int)(toupper(text[2]));
	
        /********
	*  Get the 24-bit value
        ********/
	result= (a<<16) & 0xff0000;
	result= result + ((b<<8) & 0x00ff00);
	result= result + (c & 0x0000ff);

        /********
	*  return the result.
        ********/
	return result;
}

/******************************************************************************
*
*               END OF CODE. MODULE    DSPCommand.c
*
******************************************************************************/
